package ru.yandex.solomon.gateway.www.ser;

import java.time.Duration;
import java.time.Instant;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.solomon.gateway.www.time.UiInstant;
import ru.yandex.solomon.util.time.Interval;
import ru.yandex.solomon.util.www.ser.WwwDurationSerializer;

/**
 * @author Stepan Koltsov
 */
public class WwwIntervalSerializerTest {

    @Test
    public void parseIntervalDefault() {
        Instant now = Instant.parse("2014-08-21T18:01:26.000Z");
        Interval interval = WwwIntervalSerializer.parseInterval("", "").toInterval(now);
        Interval expected = new Interval(now.minus(Duration.ofDays(1)), now);
        Assert.assertEquals(expected, interval);
    }

    @Test
    public void parseIntervalInterval() {
        Instant now = Instant.parse("2014-08-21T18:01:26.000Z");
        Interval interval = WwwIntervalSerializer.parseInterval(
                "2014-08-21T10:01:26.000Z", "2014-08-21T20:01:26.000Z").toInterval(now);
        Interval expected = new Interval(
                Instant.parse("2014-08-21T10:01:26.000Z"), Instant.parse("2014-08-21T20:01:26.000Z"));
        Assert.assertEquals(expected, interval);
    }

    @Test
    public void parseIntervalRelative() {
        Instant now = Instant.parse("2014-08-21T18:01:26.000Z");
        Interval interval = WwwIntervalSerializer.parseInterval(
                "2h", "").toInterval(now);
        Interval expected = new Interval(
                Instant.parse("2014-08-21T16:01:26.000Z"), Instant.parse("2014-08-21T18:01:26.000Z"));
        Assert.assertEquals(expected, interval);
    }

    private void testInstantParser(String expected, String actual) {
        Instant expectedInstant = Instant.parse(expected);
        Instant actualInstant = WwwIntervalSerializer.parseInstant(actual);
        Assert.assertEquals(expectedInstant, actualInstant);
    }

    @Test
    public void parseInstant() {
        testInstantParser("2014-08-21T18:01:00.000Z", "2014-08-21 22:01");
        testInstantParser("2014-08-21T18:01:02.000Z", "2014-08-21 22:01:02");
        testInstantParser("2014-08-21T18:01:02.020Z", "2014-08-21 22:01:02.020");
        testInstantParser("2014-08-21T18:01:02.020Z", "2014-08-21 18:01:02.020Z");
    }

    @Test
    public void instantToString() {
        Assert.assertEquals("2014-08-21T18:01:00.004Z",
                WwwIntervalSerializer.uiInstantToString(UiInstant.instant(Instant.parse("2014-08-21T18:01:00.004Z"))));
    }

    @Test
    public void durationToString() {
        Assert.assertEquals("0",
                WwwIntervalSerializer.uiInstantToString(UiInstant.now()));
        Assert.assertEquals("2h",
                WwwIntervalSerializer.uiInstantToString(UiInstant.relativeForward(WwwDurationSerializer.parseDuration("2h").get())));
        Assert.assertEquals("20m",
                WwwIntervalSerializer.uiInstantToString(UiInstant.relativeForward(WwwDurationSerializer.parseDuration("20m").get())));
        Assert.assertEquals("2m3s",
                WwwIntervalSerializer.uiInstantToString(UiInstant.relativeForward(WwwDurationSerializer.parseDuration("2m3s").get())));
        Assert.assertEquals("2d",
                WwwIntervalSerializer.uiInstantToString(UiInstant.relativeForward(WwwDurationSerializer.parseDuration("2d").get())));
    }
}
