package ru.yandex.solomon.gateway.cloud;

import java.time.Clock;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import com.yandex.ydb.table.TableClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;

import ru.yandex.logbroker.agent.client.Client;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.TimeUnitConverter;
import ru.yandex.solomon.config.gateway.TBillingConfig;
import ru.yandex.solomon.config.gateway.TGatewayCloudConfig;
import ru.yandex.solomon.config.protobuf.Time;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.gateway.cloud.billing.BillingEndpoint;
import ru.yandex.solomon.gateway.cloud.billing.BillingLog;
import ru.yandex.solomon.gateway.cloud.billing.BillingLogImpl;
import ru.yandex.solomon.gateway.cloud.billing.BillingResourceFetcher;
import ru.yandex.solomon.gateway.cloud.billing.BillingShardFilter;
import ru.yandex.solomon.gateway.cloud.billing.LogBillingEndpoint;
import ru.yandex.solomon.gateway.cloud.billing.ShardByNumIdResolver;
import ru.yandex.solomon.gateway.cloud.billing.ShardByNumIdResolverImpl;
import ru.yandex.solomon.gateway.cloud.billing.StockpileBillingResourceFetchManager;
import ru.yandex.solomon.gateway.cloud.billing.stockpile.StockpileBillingResourceFetcher;
import ru.yandex.solomon.gateway.cloud.billing.stockpile.StockpileUsageProviderFactory;
import ru.yandex.solomon.gateway.cloud.billing.stockpile.StockpileUsageProviderFactoryImpl;
import ru.yandex.solomon.locks.LockService;
import ru.yandex.solomon.locks.LockServiceImpl;
import ru.yandex.solomon.locks.dao.LocksDao;
import ru.yandex.solomon.locks.dao.ydb.YdbLocksDao;
import ru.yandex.solomon.selfmon.mon.DaoMetricsProxy;
import ru.yandex.solomon.spring.ConditionalOnBean;
import ru.yandex.solomon.unified.agent.UnifiedAgentClients;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
@ConditionalOnBean(TGatewayCloudConfig.class)
public class BillingContext {

    private TBillingConfig config;
    private ThreadPoolProvider threads;

    @Autowired
    public BillingContext(TGatewayCloudConfig cloudConfig, ThreadPoolProvider threads) {
        this.config = cloudConfig.getBillingConfig();
        this.threads = threads;
    }

    @Bean
    public ShardByNumIdResolverImpl shardByNumIdResolver() {
        return new ShardByNumIdResolverImpl();
    }

    @Bean
    public BillingShardFilter billingShardFilter(ShardByNumIdResolverImpl resolver) {
        return new BillingShardFilter(resolver);
    }

    @Bean
    public BillingLog billingLog(
            @Qualifier("billingUnifiedAgentIron") Client ironClient,
            @Qualifier("billingUnifiedAgentCloud") Client cloudClient,
            FeatureFlagsHolder flags,
            MetricRegistry registry
            )
    {
        return new BillingLogImpl(ironClient.newSession().build(), cloudClient.newSession().build(), flags, registry);
    }

    @Bean
    public LogBillingEndpoint billingEndpoint(ShardByNumIdResolver resolver, BillingLog log) {
        return new LogBillingEndpoint(resolver, log);
    }

    @Bean
    public StockpileBillingResourceFetcher stockpileFetcher(MetricRegistry registry) {
        var executor = threads.getExecutorService("CpuLowPriority", "ThreadPools");
        return new StockpileBillingResourceFetcher(executor, registry);
    }

    @Bean
    public StockpileUsageProviderFactory stockpileUsageProviderFactory(
        BillingResourceFetcher fetcher,
        BillingShardFilter filter)
    {
        var executor = threads.getExecutorService("CpuLowPriority", "ThreadPools");
        var timer = threads.getSchedulerExecutorService();
        return new StockpileUsageProviderFactoryImpl(
            fetcher,
            filter,
            executor,
            timer,
            config.getStockpileClustersList());
    }

    @Bean
    public StockpileBillingResourceFetchManager billingFetchManager(
        LockService lockService,
        BillingEndpoint billingEndpoint,
        StockpileUsageProviderFactory factory,
        MetricRegistry registry
        )
    {
        var executor = threads.getExecutorService("CpuLowPriority", "ThreadPools");
        var timer = threads.getSchedulerExecutorService();
        return new StockpileBillingResourceFetchManager(
            lockService,
            Clock.systemUTC(),
            timer,
            executor,
            getMillis(config.getPushInterval(), TimeUnit.MINUTES.toMillis(1)),
            billingEndpoint,
            factory,
            TimeUnit.MINUTES.toMillis(1),
            registry);
    }

    @Bean
    @Qualifier("billingUnifiedAgentIron")
    public Client billingUnifiedAgentIron(MetricRegistry registry) {
        return UnifiedAgentClients.makeClient(config.getUnifiedAgentClient(0), threads, registry);
    }

    @Bean
    @Qualifier("billingUnifiedAgentCloud")
    public Client billingUnifiedAgentCloud(MetricRegistry registry) {
        return UnifiedAgentClients.makeClient(config.getUnifiedAgentClient(1), threads, registry);
    }

    @Bean
    @Primary
    LockService lockService(ThreadPoolProvider tpProvider, LocksDao dao, MetricRegistry registry)
    {
        ScheduledExecutorService timer = tpProvider.getSchedulerExecutorService();
        return new LockServiceImpl(HostUtils.getFqdn(), dao, Clock.systemUTC(), timer, registry);
    }

    @Bean
    @Primary
    LocksDao locksDao(
        TableClient tableClient,
        MetricRegistry registry,
        @Qualifier("configKikimrRootPath") String rootPathStr)
    {
        var rootPath = rootPathStr + "/CloudIndex/V1";
        YdbLocksDao dao = new YdbLocksDao(tableClient, Clock.systemUTC(), rootPath);
        return DaoMetricsProxy.of(dao, LocksDao.class, registry);
    }

    private long getMillis(Time time, long defaultMillis) {
        if (time.getValue() == 0) {
            return defaultMillis;
        }

        return TimeUnitConverter.protoToUnit(time.getUnit()).toMillis(time.getValue());
    }
}
