package ru.yandex.solomon.gateway.cloud;

import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.ExecutorService;

import com.google.common.net.HostAndPort;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.cloud.resourcemanager.GrpcResourceManagerClient;
import ru.yandex.cloud.resourcemanager.ResourceManagerClient;
import ru.yandex.cloud.resourcemanager.ResourceManagerClientOptions;
import ru.yandex.cloud.session.SessionClient;
import ru.yandex.cloud.session.SessionClientOptions;
import ru.yandex.cloud.session.grpc.GrpcSessionClient;
import ru.yandex.cloud.token.IamOauthClient;
import ru.yandex.cloud.token.IamOauthClientOptions;
import ru.yandex.cloud.token.IamTokenClient;
import ru.yandex.cloud.token.IamTokenClientOptions;
import ru.yandex.cloud.token.Jwt;
import ru.yandex.cloud.token.grpc.GrpcIamTokenClient;
import ru.yandex.cloud.token.http.HttpIamOauthClient;
import ru.yandex.discovery.DiscoveryServices;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolver;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolverImpl;
import ru.yandex.solomon.cloud.resource.resolver.CloudProjectNamesUpdater;
import ru.yandex.solomon.cloud.resource.resolver.FolderResolver;
import ru.yandex.solomon.cloud.resource.resolver.FolderResolverImpl;
import ru.yandex.solomon.config.gateway.TGatewayCloudConfig;
import ru.yandex.solomon.config.protobuf.frontend.TAuthConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.YdbClientsContext;
import ru.yandex.solomon.gateway.cloud.unified_agent.CloudConfigController;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.spring.ConditionalOnBean;
import ru.yandex.solomon.util.SolomonEnv;

import static com.google.common.base.Preconditions.checkArgument;


/**
 * @author Sergey Polovko
 */
@Configuration
@Import({
        CloudProjectNamesUpdater.class,
        BillingContext.class,
        YdbClientsContext.class,
        SearchContext.class,
})
@ConditionalOnBean(TGatewayCloudConfig.class)
public class GatewayCloudContext {
    private static final Logger logger = LoggerFactory.getLogger(GatewayCloudContext.class);

    private static final Duration DEFAULT_CONNECT_TIMEOUT = Duration.ofSeconds(5);
    private static final Duration DEFAULT_REQUEST_TIMEOUT = Duration.ofSeconds(30);

    @Bean
    CloudConfigController cloudConfigController(Optional<TGatewayCloudConfig> config) {
        if (config.isEmpty()) {
            return null;
        }

        return new CloudConfigController(config.get().getUnifiedAgentInstallerConfig());
    }

    @Bean(name = "iamTokenProvider")
    TokenProvider tokenProvider(IamTokenClient iamTokenClient, TAuthConfig authConfig, ThreadPoolProvider threads) {
        if (!authConfig.hasIamConfig()) {
            logger.warn("using fake token provider");
            return TokenProvider.of("fake-token");
        }

        TAuthConfig.TIamConfig config = authConfig.getIamConfig();

        Path keyFilePath = Path.of(config.getKeyAuth().getKeyPath());
        if (SolomonEnv.DEVELOPMENT.isActive() && !Files.exists(keyFilePath)) {
            // only in development environment allowed to make fake token provide
            logger.warn("using fake token provider");
            return TokenProvider.of("fake-token");
        }

        var jwtBuilder = Jwt.newBuilder()
                .withAccountId(config.getKeyAuth().getAccountId())
                .withKeyId(config.getKeyAuth().getKeyId())
                .withPrivateKey(keyFilePath)
                .withTtl(Duration.ofHours(1));

        return TokenProvider.iam(iamTokenClient, jwtBuilder, threads.getSchedulerExecutorService());
    }

    @Bean
    IamTokenClient iamTokenClient(TAuthConfig authConfig, ThreadPoolProvider threads) {
        var config = authConfig.getIamConfig();

        ExecutorService executor = threads.getExecutorService(
                config.getThreadPoolName(),
                "CloudConfig.IamTokenClient.ThreadPoolName");

        HostAndPort address = DiscoveryServices.resolve(config.getTokenServiceAddresses()).get(0);
        return new GrpcIamTokenClient(IamTokenClientOptions.forAddress(address.getHost(), address.getPort())
                .withHandlerExecutor(executor)
                .withUserAgent("SolomonGateway"));
    }

    @Bean
    SessionClient sessionClient(
            TGatewayCloudConfig cloudConfig,
            @Qualifier("iamTokenProvider") TokenProvider tokenProvider,
            ThreadPoolProvider threads, MetricRegistry registry)
    {
        var config = cloudConfig.getSessionClient();
        ExecutorService executor = threads.getExecutorService(
                config.getThreadPoolName(),
                "CloudConfig.SessionClient.ThreadPoolName");
        return new GrpcSessionClient(SessionClientOptions.forAddress(config.getHost(), config.getPort())
                .withOauthEndpoint(cloudConfig.getIamOauthClient().getEndpoint())
                .withUserAgent("SolomonGateway")
                .withConnectTimeout(DEFAULT_CONNECT_TIMEOUT)
                .withRequestTimeout(DEFAULT_REQUEST_TIMEOUT)
                .withHandlerExecutor(executor)
                .withTokenProvider(tokenProvider)
                .withRegistry(registry));
    }

    @Bean
    IamOauthClient iamOauthClient(
            TGatewayCloudConfig cloudConfig,
            TAuthConfig authConfig,
            ThreadPoolProvider threads,
            SecretProvider secrets)
    {
        var config = cloudConfig.getIamOauthClient();
        ExecutorService executor = threads.getExecutorService(
                config.getThreadPoolName(),
                "CloudConfig.IamOauthClient.ThreadPoolName");

        checkArgument(authConfig.hasOpenIdConfig(), "empty AuthConfig.OpenIdConfig section");
        var openIdConfig = authConfig.getOpenIdConfig();

        Optional<String> clientSecretO = secrets.getSecret(openIdConfig.getClientSecret());
        String clientSecret = SolomonEnv.DEVELOPMENT.isActive()
                ? clientSecretO.orElse("fake")
                : clientSecretO.orElseThrow(() -> new RuntimeException("cannot create IamOauthClient without client secret: " + openIdConfig.getClientSecret()));

        return new HttpIamOauthClient(IamOauthClientOptions.forEndpoint(config.getEndpoint())
                .withConnectTimeout(DEFAULT_CONNECT_TIMEOUT)
                .withRequestTimeout(DEFAULT_REQUEST_TIMEOUT)
                .withClientId(openIdConfig.getClientId())
                .withClientSecret(clientSecret)
                .withHandlerExecutor(executor));
    }

    @Bean
    ResourceManagerClient resourceManagerClient(
            TGatewayCloudConfig cloudConfig,
            @Qualifier("iamTokenProvider") TokenProvider tokenProvider,
            ThreadPoolProvider threads,
            MetricRegistry registry)
    {
        var config = cloudConfig.getResourceManagerClient();
        ExecutorService executor = threads.getExecutorService(
                config.getThreadPoolName(),
                "CloudConfig.ResourceManagerClient.ThreadPoolName");
        return new GrpcResourceManagerClient(ResourceManagerClientOptions.forAddress(config.getHost(), config.getPort())
                .withUserAgent("SolomonGateway")
                .withConnectTimeout(DEFAULT_CONNECT_TIMEOUT)
                .withRequestTimeout(DEFAULT_REQUEST_TIMEOUT)
                .withHandlerExecutor(executor)
                .withTokenProvider(tokenProvider)
                .withRegistry(registry)
                .withResolveExistingOnly(config.getResolveExistingOnly()));
    }

    @Bean
    CloudByFolderResolver cloudByFolderResolver(ResourceManagerClient resourceManagerClient) {
        return new CloudByFolderResolverImpl(resourceManagerClient);
    }

    @Bean
    FolderResolver folderResolver(
        SolomonConfHolder confHolder,
        CloudByFolderResolver cloudByFolderResolver)
    {
        return new FolderResolverImpl(confHolder, cloudByFolderResolver);
    }
}
