package ru.yandex.solomon.gateway.cloud;

import java.time.Clock;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.ExecutorService;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.logbroker.agent.client.Client;
import ru.yandex.logbroker.agent.client.SessionMetrics;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.client.NotificationApi;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolver;
import ru.yandex.solomon.conf.db3.MonitoringDashboardsDao;
import ru.yandex.solomon.config.TimeUnitConverter;
import ru.yandex.solomon.config.gateway.TGatewayCloudConfig;
import ru.yandex.solomon.config.gateway.TSearchConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.gateway.cloud.search.AlertFetcher;
import ru.yandex.solomon.gateway.cloud.search.ChannelFetcher;
import ru.yandex.solomon.gateway.cloud.search.DashboardFetcher;
import ru.yandex.solomon.gateway.cloud.search.ReindexScheduler;
import ru.yandex.solomon.gateway.cloud.search.ResourceFetcher;
import ru.yandex.solomon.gateway.cloud.search.SearchEventSink;
import ru.yandex.solomon.gateway.cloud.search.SearchEventSinkImpl;
import ru.yandex.solomon.locks.LockService;
import ru.yandex.solomon.spring.ConditionalOnBean;
import ru.yandex.solomon.unified.agent.UnifiedAgentClients;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
@ConditionalOnBean(TGatewayCloudConfig.class)
public class SearchContext {

    private final TSearchConfig config;
    private final ThreadPoolProvider threads;
    private final ExecutorService executor;
    private final MetricRegistry registry;
    private final Clock clock;

    @Autowired
    public SearchContext(TGatewayCloudConfig cloudConfig, ThreadPoolProvider threads, MetricRegistry registry) {
        this.config = cloudConfig.getSearchConfig();
        this.threads = threads;
        this.executor = threads.getExecutorService(config.getThreadPoolName(), "ThreadPoolName");
        this.registry = registry;
        this.clock = Clock.systemUTC();
    }

    @Bean
    @Qualifier("searchEventSink")
    public Client searchUnifiedAgent() {
        return UnifiedAgentClients.makeClient(config.getUnifiedAgentClient(), threads, registry);
    }

    @Bean
    public SearchEventSinkImpl searchUnifiedAgentSession(@Qualifier("searchEventSink") Client client) {
        var session = client.newSession()
                .setMetrics(new SessionMetrics(registry.subRegistry("session", "search-events")))
                .build();

        return new SearchEventSinkImpl(session);
    }

    @Bean
    public DashboardFetcher dashboardFetcher(MonitoringDashboardsDao dao, CloudByFolderResolver cloudResolver) {
        return new DashboardFetcher(dao, cloudResolver);
    }

    @Bean
    public AlertFetcher alertFetcher(SolomonConfHolder confHolder, AlertApi alertApi) {
        return new AlertFetcher(() -> confHolder.getConfOrThrow().projects(), alertApi, executor);
    }

    @Bean
    public ChannelFetcher channelFetcher(SolomonConfHolder confHolder, NotificationApi notificationApi) {
        return new ChannelFetcher(() -> confHolder.getConfOrThrow().projects(), notificationApi, executor);
    }

    @Bean
    public ReindexScheduler reindexScheduler(LockService lockService, List<ResourceFetcher> fetchers, SearchEventSink sink) {
        var lock = lockService.distributedLock("SearchReindexMaster");
        var timer = threads.getSchedulerExecutorService();
        var period = Duration.ofMillis(TimeUnitConverter.millis(config.getReindexInterval()));
        var delay = Duration.ofMillis(TimeUnitConverter.millis(config.getInitialReindexDelay()));
        return new ReindexScheduler(lock, period, delay, fetchers, sink, executor, timer, clock, registry);
    }
}
