package ru.yandex.solomon.gateway.cloud.api;

import java.security.GeneralSecurityException;
import java.util.Objects;
import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import com.google.common.base.Strings;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.solomon.common.RequestProducer;

/**
 * @author Vladimir Gordiychuk
 */
public class RequestProducerResolverImpl implements RequestProducerResolver {
    private static final Pattern PATTERN = Pattern.compile("(?<signature>.*):ui:(?<staff>.*):(?<time>\\d*)");
    private static final Logger logger = LoggerFactory.getLogger(RequestProducerResolverImpl.class);
    private final SecretKeySpec key;

    public RequestProducerResolverImpl(String secret) {
        this.key = new SecretKeySpec(secret.getBytes(CharsetUtils.UTF8_CHARSET), "");
    }

    @Override
    public RequestProducer resolve(@Nullable String header) {
        if (Strings.isNullOrEmpty(header)) {
            return RequestProducer.USER;
        }

        var matcher = PATTERN.matcher(header);
        if (!matcher.matches()) {
            return RequestProducer.USER;
        }

        String staffLogin = matcher.group("staff");
        String timestamp = matcher.group("time");
        String expectedSign = sign(staffLogin, timestamp);

        if (!Objects.equals(expectedSign, matcher.group("signature"))) {
            logger.warn("signature mismatch " + header);
            return RequestProducer.USER;
        }

        if (Strings.isNullOrEmpty(staffLogin)) {
            return RequestProducer.SYSTEM;
        }

        return RequestProducer.STAFF;
    }

    String sign(String staffLogin, String timestamp) {
        String message = "ui:" + staffLogin + ":" + timestamp;
        try {
            Mac mac = Mac.getInstance("HmacSHA1");
            mac.init(key);
            return Base64.toBase64String(mac.doFinal(message.getBytes(CharsetUtils.UTF8_CHARSET)));
        } catch (GeneralSecurityException e) {
            throw new RuntimeException(e);
        }
    }
}
