package ru.yandex.solomon.gateway.cloud.billing;

import java.util.UUID;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.logbroker.agent.client.Session;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class BillingLogImpl implements BillingLog {
    private static final Logger billingLog = LoggerFactory.getLogger("billing");
    private static final Logger logger = LoggerFactory.getLogger(BillingLogImpl.class);

    private final FeatureFlagsHolder flags;
    private final ObjectMapper mapper;
    private final Rate writeRate;
    private final Rate writePoints;
    private final Rate readPoints;
    private final Rate storePoints;
    private Session uaSessionIron;
    private Session uaSessionCloud;

    public BillingLogImpl(Session uaSessionIron, Session uaSessionCloud, FeatureFlagsHolder flags, MetricRegistry registry) {
        this.uaSessionIron = uaSessionIron;
        this.uaSessionCloud = uaSessionCloud;
        this.flags = flags;
        this.mapper = new ObjectMapper();
        this.writeRate = registry.rate("billing.log.write");
        this.writePoints = registry.rate("billing.write.points");
        this.readPoints = registry.rate("billing.read.points");
        this.storePoints = registry.rate("billing.store.points");
    }

    @Override
    public void writePoints(String cloudId, String folderId, MetricType type, long count, long startSec, long finishSec) {
        if (count == 0) {
            return;
        }

        String sku = schema(type, ".write");
        writeDelta(cloudId, folderId, sku, count, startSec, finishSec);
        writePoints.add(count);
    }

    @Override
    public void readPoints(String cloudId, String folderId, MetricType type, long count, long startSec, long finishSec) {
        if (count == 0) {
            return;
        }

        String sku = schema(type, ".read");
        writeDelta(cloudId, folderId, sku, count, startSec, finishSec);
        readPoints.add(count);
    }

    @Override
    public void storePoints(String cloudId, String folderId, MetricType type, long count, long startSec, long finishSec) {
        if (count == 0) {
            return;
        }
        // TODO: enable when business will be ready count user reads
        // String sku = schema(type, ".store");
        // writeDelta(cloudId, folderId, sku, count, startedAt, finishAt);
        storePoints.add(count);
    }

    private void writeDelta(String cloudId, String folderId, String sku, long count, long startSec, long finishSec) {
        var usage = mapper.createObjectNode();
        usage.put("quantity", count);
        usage.put("type", "delta");
        usage.put("unit", "point");
        usage.put("start", startSec);
        usage.put("finish", finishSec);

        var metric = mapper.createObjectNode();
        metric.put("id", UUID.randomUUID().toString());
        metric.put("schema", sku);
        metric.put("cloud_id", cloudId);
        metric.put("folder_id", folderId);
        metric.set("usage", usage);
        metric.set("tags", mapper.createObjectNode());
        metric.put("source_wt", finishSec);
        metric.put("source_id", HostUtils.getFqdn());
        metric.put("version", "v1alpha1");
        try {
            var json = mapper.writeValueAsString(metric);
            if(flags.hasFlag(FeatureFlag.USE_UNIFIED_AGENT_FOR_BILLING, cloudId)){
                if (flags.hasFlag(FeatureFlag.USE_CLOUD_LOGBROKER_FOR_BILLING, cloudId)) {
                    writeToUA(uaSessionCloud, json);
                    return;
                }
                writeToUA(uaSessionIron, json);
                return;
            }
            log(json);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    private static String schema(MetricType type, String postfix) {
        return "monitoring.point." + type.name().toLowerCase() + postfix;
    }

    protected void log(String json) {
        billingLog.info(json);
        writeRate.inc();
    }

    protected void writeToUA(Session session, String json){
        logger.debug("send: {}", json);
        session.send(json);
        writeRate.inc();
    }
}
