package ru.yandex.solomon.gateway.cloud.billing;

import java.time.Clock;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.WillCloseWhenClosed;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.gateway.cloud.billing.stockpile.StockpileUsage;
import ru.yandex.solomon.gateway.cloud.billing.stockpile.StockpileUsageProvider;
import ru.yandex.solomon.gateway.cloud.billing.stockpile.StockpileUsageProviderFactory;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.LockService;
import ru.yandex.solomon.locks.LockSubscriber;
import ru.yandex.solomon.locks.UnlockReason;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileBillingResourceFetchManager implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(StockpileBillingResourceFetchManager.class);

    private final Clock clock;
    private final ScheduledExecutorService timer;
    private final ExecutorService executor;
    private final long pushIntervalMillis;
    private final BillingEndpoint billingEndpoint;
    private final StockpileUsageProviderFactory stockpileFactory;
    private final DistributedLock distributedLock;
    private final AtomicReference<State> state = new AtomicReference<>(new Slave());
    private final long leaseMillis;
    private volatile boolean closed;

    private final AsyncMetrics pushMetrics;

    public StockpileBillingResourceFetchManager(
        LockService lockService,
        Clock clock,
        ScheduledExecutorService timer,
        ExecutorService executor,
        long pushIntervalMillis,
        BillingEndpoint billingEndpoint,
        StockpileUsageProviderFactory stockpileFactory,
        long leaseMillis,
        MetricRegistry registry)
    {
        this.clock = clock;
        this.timer = timer;
        this.executor = executor;
        this.pushIntervalMillis = pushIntervalMillis;
        this.billingEndpoint = billingEndpoint;
        this.stockpileFactory = stockpileFactory;
        this.pushMetrics = new AsyncMetrics(registry, "billing.push");
        this.distributedLock = lockService.distributedLock("BillingMaster");
        this.leaseMillis = leaseMillis;
        aquireMasterLock();
    }

    private void aquireMasterLock() {
        this.distributedLock.acquireLock(new LockSubscriber() {
            @Override
            public boolean isCanceled() {
                return closed;
            }

            @Override
            public void onLock(long seqNo) {
                long now = clock.millis();
                State prev = state.get();
                stockpileFactory.create()
                        .whenComplete((provider, e) -> {
                            if (e != null) {
                                logger.error("unable to create stockpile usage provider", e);
                                distributedLock.unlock();
                            } else {
                                var master = new Master(seqNo, now, provider);
                                if (state.compareAndSet(prev, master)) {
                                    prev.close();
                                } else {
                                    master.close();
                                }
                            }
                        });
            }

            @Override
            public void onUnlock(UnlockReason reason) {
                state.getAndSet(new Slave()).close();
                if (!closed) {
                    executor.submit(() -> aquireMasterLock());
                }
            }
        }, leaseMillis, TimeUnit.MILLISECONDS);
    }

    @Override
    public void close() {
        closed = true;
        state.get().close();
    }

    private interface State extends AutoCloseable {
        @Override
        void close();
    }

    private class Master implements State {
        private final long seqNo;
        @WillCloseWhenClosed
        private final StockpileUsageProvider stockpile;

        private long lastFetchMillis;
        private volatile boolean closed;
        private volatile Future scheduled;

        public Master(long seqNo, long nowMillis, @WillCloseWhenClosed StockpileUsageProvider stockpile) {
            this.seqNo = seqNo;
            this.stockpile = stockpile;
            this.lastFetchMillis = nowMillis;
            this.scheduleNextFetch();
        }

        private void scheduleNextFetch() {
            long past = clock.millis() - lastFetchMillis;
            long delay = pushIntervalMillis - past;
            scheduled = timer.schedule(() -> executor.execute(this::runScheduled), delay, TimeUnit.MILLISECONDS);
            if (closed) {
                scheduled.cancel(false);
            }
        }

        public void runScheduled() {
            if (closed) {
                return;
            }

            push(stockpile.getUsage())
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        logger.error("billing push failed", e);
                    }

                    scheduleNextFetch();
                });
        }

        private CompletableFuture<Void> push(StockpileUsage usage) {
            long now = clock.millis();
            var interval = Interval.millis(lastFetchMillis, now);
            lastFetchMillis = now;

            var future = billingEndpoint.push(usage, interval, seqNo);
            pushMetrics.forFuture(future);
            return future;
        }

        @Override
        public void close() {
            closed = true;
            scheduled.cancel(false);
            stockpile.close();
        }
    }

    private static class Slave implements State {
        @Override
        public void close() {
            // no op
        }
    }
}
