package ru.yandex.solomon.gateway.cloud.billing.stockpile;

import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;

import com.google.common.net.HostAndPort;
import com.google.common.net.HttpHeaders;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.encode.spack.MetricSpackDecoder;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.gateway.cloud.billing.BillingResourceFetcher;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileBillingResourceFetcher implements BillingResourceFetcher, AutoCloseable {
    private final HttpClient httpClient;
    private final Labels endpoint = Labels.of("endpoint", "Stockpile/resourceUsage/metrics");
    private final AsyncMetrics asyncMetrics;
    private final MetricRegistry registry;

    public StockpileBillingResourceFetcher(ExecutorService executor, MetricRegistry registry) {
        this.asyncMetrics = new AsyncMetrics(registry, "http.client.call", endpoint);
        this.registry = registry;
        this.httpClient = HttpClient.newBuilder()
            .connectTimeout(Duration.ofSeconds(10))
            .followRedirects(HttpClient.Redirect.NEVER)
            .version(HttpClient.Version.HTTP_1_1)
            .executor(executor)
            .build();
    }

    @Override
    public CompletableFuture<Void> fetch(HostAndPort address, MetricConsumer consumer) {
        try {
            var request = createRequest(address);
            var future = httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofByteArray())
                .thenAccept(response -> {
                    registry.rate("http.client.call.status", endpoint.add("status", String.valueOf(response.statusCode()))).inc();
                    if (response.statusCode() != 200) {
                        throw new IllegalStateException("not 200: " + response);
                    }

                    var body = response.body();
                    registry.rate("http.client.call.inboundBytes", endpoint).add(body.length);
                    new MetricSpackDecoder().decode(body, consumer);
                });
            asyncMetrics.forFuture(future);
            return future;
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private HttpRequest createRequest(HostAndPort address) {
        return HttpRequest.newBuilder()
            .uri(URI.create("http://" + address.toString() + "/resourceUsage/metrics"))
            .GET()
            .header(HttpHeaders.ACCEPT, "application/x-solomon-spack")
            .header(HttpHeaders.ACCEPT_ENCODING, "zstd,lz4,gzip,deflate")
            .timeout(Duration.ofSeconds(10))
            .build();
    }

    @Override
    public void close() {

    }
}
