package ru.yandex.solomon.gateway.cloud.billing.stockpile;

import java.util.ArrayList;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.IntPredicate;

import javax.annotation.Nullable;

import com.google.common.net.HostAndPort;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.gateway.cloud.billing.BillingResourceFetcher;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileHostUsageProvider implements StockpileUsageProvider {
    private static final Logger logger = LoggerFactory.getLogger(StockpileHostUsageProvider.class);
    private final HostAndPort address;
    private final BillingResourceFetcher fetcher;
    private final IntPredicate ownerShardIdFilter;
    private final ScheduledExecutorService timer;
    @Nullable
    private volatile StockpileUsage prev;
    private final ArrayBlockingQueue<StockpileUsage> queue = new ArrayBlockingQueue<>(100);

    private volatile boolean closed;
    private volatile Future<?> future;

    public StockpileHostUsageProvider(
        HostAndPort address,
        BillingResourceFetcher fetcher,
        IntPredicate ownerShardIdFilter,
        ScheduledExecutorService timer)
    {
        this.address = address;
        this.fetcher = fetcher;
        this.ownerShardIdFilter = ownerShardIdFilter;
        this.timer = timer;
        scheduleNext();
    }

    @Override
    public StockpileUsage getUsage() {
        var items = new ArrayList<StockpileUsage>(queue.size());
        queue.drainTo(items);
        if (items.isEmpty()) {
            return new StockpileUsage(0);
        } else if (items.size() == 1) {
            return items.get(0);
        }

        StockpileUsage total = new StockpileUsage(items.get(0).usageByKey.size());
        for (StockpileUsage usage : items) {
            total.combineDelta(usage);
        }
        return total;
    }


    private void runScheduledFetch() {
        var parser = new StockpileResourceUsageParser(ownerShardIdFilter);
        future = fetcher.fetch(address, parser)
            .handle((ignore, e) -> {
                if (e != null) {
                    logger.warn("fetch resource usage by host {} failed with error", address, e);
                    prev = null;
                } else {
                    var actual = parser.getUsage();
                    var delta = calculateDelta(actual, prev);
                    if (queue.offer(delta)) {
                        prev = actual;
                    }
                }

                scheduleNext();
                return null;
            });
    }

    private void scheduleNext() {
        long delayMillis = ThreadLocalRandom.current().nextLong(10_000, 15_000);
        future = timer.schedule(this::runScheduledFetch, delayMillis, TimeUnit.MILLISECONDS);
        if (closed) {
            future.cancel(false);
            queue.clear();
        }
    }

    private StockpileUsage calculateDelta(StockpileUsage actual, @Nullable StockpileUsage prev) {
        StockpileUsage delta = new StockpileUsage(actual.usageByKey.size());
        for (var entry : actual.usageByKey.long2ObjectEntrySet()) {
            var key = entry.getLongKey();
            var actualUsage = entry.getValue();

            // fetched list
            var deltaUsage = delta.getUsage(key);
            deltaUsage.storeMetrics = actualUsage.storeMetrics;
            deltaUsage.storeRecords = actualUsage.storeRecords;

            var prevUsage = prev != null ? prev.getUsageOrNull(key) : null;
            if (prevUsage != null) {
                deltaUsage.readMetrics += diff(actualUsage.readMetrics, prevUsage.readMetrics);
                deltaUsage.readRecords += diff(actualUsage.readRecords, prevUsage.readRecords);
                deltaUsage.writeMetrics += diff(actualUsage.writeMetrics, prevUsage.writeMetrics);
                deltaUsage.writeRecords += diff(actualUsage.writeRecords, prevUsage.writeRecords);
            }
        }
        return delta;
    }

    private long diff(long actual, long prev) {
        return actual > prev ? actual - prev : 0;
    }

    @Override
    public void close() {
        closed = true;
        future.cancel(false);
        queue.clear();
    }

    @Override
    public String toString() {
        return "StockpileHostUsageProvider{" + address + '}';
    }
}
