package ru.yandex.solomon.gateway.cloud.billing.stockpile;

import java.util.function.IntPredicate;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.histogram.HistogramSnapshot;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.solomon.common.RequestProducer;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileResourceUsageParser implements MetricConsumer {
    private static final Logger logger = LoggerFactory.getLogger(StockpileResourceUsageParser.class);

    private final IntPredicate userShardFilter;
    private StockpileUsage summary = new StockpileUsage();
    private ru.yandex.solomon.model.protobuf.MetricType usageMetricType;
    private boolean ignore;
    private int ownerShardId;
    private String metric;
    private String producer;

    public StockpileResourceUsageParser(IntPredicate userShardFilter) {
        this.userShardFilter = userShardFilter;
    }

    @Override
    public void onStreamBegin(int countHint) {
    }

    @Override
    public void onStreamEnd() {
    }

    @Override
    public void onCommonTime(long tsMillis) {
    }

    @Override
    public void onMetricBegin(MetricType type) {
        ignore = !(type == MetricType.RATE || type == MetricType.IGAUGE);
    }

    @Override
    public void onMetricEnd() {
        ignore = false;
        usageMetricType = null;
        metric = null;
        producer = null;
        ownerShardId = 0;
    }

    @Override
    public void onLabelsBegin(int countHint) {
    }

    @Override
    public void onLabelsEnd() {
        if (ignore) {
            return;
        }

        ignore = usageMetricType == null || StringUtils.isEmpty(metric) || ownerShardId == 0;
    }

    @Override
    public void onLabel(Label label) {
        onLabel(label.getKey(), label.getValue());
    }

    @Override
    public void onLabel(String key, String value) {
        if (ignore) {
            return;
        }

        try {
            switch (key) {
                case "ownerId": {
                    if ("total".equals(value) || "0".equals(value) || value.startsWith("-")) {
                        ignore = true;
                        return;
                    }

                    ownerShardId = Integer.parseUnsignedInt(value);
                    return;
                }
                case "type":
                case "kind": {
                    if ("total".equals(value)) {
                        ignore = true;
                        return;
                    }

                    usageMetricType = ru.yandex.solomon.model.protobuf.MetricType.valueOf(value);
                    return;
                }
                case "level":
                    ignore = !"total".equals(value);
                    return;
                case "projectId":
                    ignore = !"SOLOMON".equals(value);
                    return;
                case "sensor":
                    metric = value;
                    return;
                case "producer":
                    producer = value;
                    return;
                case "host":
                    return;
                default:
                    ignore = true;
            }
        } catch (Throwable e) {
            ignore = true;
            logger.warn("unable to parse {}={}, ignore it", key, value, e);
        }
    }

    @Override
    public void onDouble(long tsMillis, double value) {
    }

    @Override
    public void onLong(long tsMillis, long value) {
        if (ignore) {
            return;
        }

        if (userShardFilter.test(ownerShardId) && (
                producer == null ||
                RequestProducer.REQUEST_PRODUCER_UNSPECIFIED.name().equals(producer) ||
                "UNKNOWN_PRODUCER".equals(producer)))
        {
            switch (metric) {
                case "stockpile.write.sensors.rate":
                    getShardUsage().writeMetrics = value;
                    return;
                case "stockpile.write.records.rate":
                    getShardUsage().writeRecords = value;
                    return;
                case "stockpile.host.sensors.count":
                    getShardUsage().storeMetrics = value;
                    return;
                case "stockpile.host.record.count":
                    getShardUsage().storeRecords = value;
                    return;
            }
        }

        if (RequestProducer.USER.name().equals(producer)) {
            switch (metric) {
                case "stockpile.read.sensors.rate":
                    getShardUsage().readMetrics += value;
                    break;
                case "stockpile.read.records.rate":
                    getShardUsage().readRecords += value;
                    break;
            }
        }
    }

    private StockpileUsage.Usage getShardUsage() {
        return summary.getUsage(usageMetricType, ownerShardId);
    }

    @Override
    public void onHistogram(long tsMillis, HistogramSnapshot snapshot) {
    }

    public StockpileUsage getUsage() {
        return summary;
    }
}
