package ru.yandex.solomon.gateway.cloud.billing.stockpile;

import javax.annotation.Nullable;

import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.solomon.model.protobuf.MetricType;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileUsage {
    public final Long2ObjectMap<Usage> usageByKey;

    public StockpileUsage() {
        this(32);
    }

    public StockpileUsage(int capacity) {
        usageByKey = new Long2ObjectOpenHashMap<>(capacity);
    }

    @SuppressWarnings("ProtocolBufferOrdinal")
    public static long key(MetricType type, int ownerShardId) {
        long result = (long) ownerShardId << 32;
        result |= (long) type.ordinal();
        return result;
    }

    public static MetricType type(long key) {
        int index = (int) (key & 0xffffL);
        return MetricType.values()[index];
    }

    public static int ownerShardId(long key) {
        return (int) (key >> 32);
    }

    public Usage getUsage(MetricType type, int ownerShardId) {
        return getUsage(key(type, ownerShardId));
    }

    public Usage getUsage(long key) {
        var usage = usageByKey.get(key);
        if (usage == null) {
            usage = new Usage();
            usageByKey.put(key, usage);
        }

        return usage;
    }

    @Nullable
    public Usage getUsageOrNull(long key) {
        return usageByKey.get(key);
    }

    @Nullable
    public Usage getUsageOrNull(MetricType type, int ownerShardId) {
        return usageByKey.get(key(type, ownerShardId));
    }

    public StockpileUsage combine(StockpileUsage delta) {
        for (var entry : delta.usageByKey.long2ObjectEntrySet()) {
            getUsage(entry.getLongKey()).add(entry.getValue());
        }
        return this;
    }

    public StockpileUsage max(StockpileUsage usage) {
        for (var entry : usage.usageByKey.long2ObjectEntrySet()) {
            getUsage(entry.getLongKey()).max(entry.getValue());
        }
        return this;
    }

    public StockpileUsage combineDelta(StockpileUsage delta) {
        for (var entry : delta.usageByKey.long2ObjectEntrySet()) {
            getUsage(entry.getLongKey()).addDelta(entry.getValue());
        }
        return this;
    }

    public static class Usage extends DefaultObject {
        public long writeRecords;
        public long writeMetrics;

        public long readRecords;
        public long readMetrics;

        public long storeRecords;
        public long storeMetrics;

        public void add(Usage delta) {
            this.writeRecords += delta.writeRecords;
            this.writeMetrics += delta.writeMetrics;

            this.readRecords += delta.readRecords;
            this.readMetrics += delta.readMetrics;

            this.storeRecords += delta.storeRecords;
            this.storeMetrics += delta.storeMetrics;
        }

        public void max(Usage usage) {
            this.writeRecords = Math.max(this.writeRecords, usage.writeRecords);
            this.writeMetrics = Math.max(this.writeMetrics, usage.writeMetrics);

            this.readRecords = Math.max(this.readRecords, usage.readRecords);
            this.readMetrics = Math.max(this.readMetrics, usage.readMetrics);

            this.storeRecords = Math.max(this.storeRecords, usage.storeRecords);
            this.storeMetrics = Math.max(this.storeMetrics, usage.storeMetrics);
        }

        public void addDelta(Usage usage) {
            this.writeRecords += usage.writeRecords;
            this.writeMetrics += usage.writeMetrics;

            this.readRecords += usage.readRecords;
            this.readMetrics += usage.readMetrics;

            this.storeRecords = Math.max(this.storeRecords, usage.storeRecords);
            this.storeMetrics = Math.max(this.storeMetrics, usage.storeMetrics);
        }
    }

    @Override
    public String toString() {
        return "StockpileUsage{" +
            "size=" + usageByKey.size() +
            '}';
    }
}
