package ru.yandex.solomon.gateway.cloud.billing.stockpile;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.IntPredicate;

import com.google.common.net.HostAndPort;

import ru.yandex.discovery.DiscoveryService;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.config.gateway.TStockpileCluster;
import ru.yandex.solomon.gateway.cloud.billing.BillingResourceFetcher;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileUsageProviderFactoryImpl implements StockpileUsageProviderFactory {
    private final BillingResourceFetcher fetcher;
    private final IntPredicate shardFilter;
    private final ExecutorService executor;
    private final ScheduledExecutorService timer;
    private final List<TStockpileCluster> cluster;

    public StockpileUsageProviderFactoryImpl(
        BillingResourceFetcher fetcher,
        IntPredicate shardFilter,
        ExecutorService executor,
        ScheduledExecutorService timer,
        List<TStockpileCluster> cluster)
    {
        this.shardFilter = shardFilter;
        this.fetcher = fetcher;
        this.executor = executor;
        this.timer = timer;
        this.cluster = cluster;
    }

    @Override
    public CompletableFuture<StockpileUsageProvider> create() {
        if (cluster.isEmpty()) {
            return CompletableFuture.completedFuture(new FakeStockpileUsageProvider());
        }

        return cluster.stream()
            .map(cluster -> DiscoveryService.async().resolve(cluster.getAddressesList())
                .thenApply(hosts -> hosts.stream()
                    .map(this::createHost)
                    .collect(collectingAndThen(toList(), list -> createDc(cluster.getName(), list)))))
            .collect(collectingAndThen(toList(), CompletableFutures::allOf))
            .thenApply(this::createCrossDc);
    }

    private StockpileUsageProvider createCrossDc(List<StockpileUsageProvider> dc) {
        if (dc.size() == 1) {
            return dc.get(0);
        }

        return new StockpileCrossDcUsageProvider(dc);
    }

    private StockpileUsageProvider createDc(String name, List<StockpileUsageProvider> hosts) {
        if (hosts.size() == 1) {
            return hosts.get(0);
        }

        return new StockpileDcUsageProvider(name, hosts);
    }

    private StockpileUsageProvider createHost(HostAndPort address) {
        return new StockpileHostUsageProvider(
            address,
            fetcher,
            shardFilter,
            timer);
    }
}
