package ru.yandex.solomon.gateway.cloud.search;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Consumer;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public abstract class AbstractAlertingFetcher<T> implements ResourceFetcher {
    protected static final int PAGE_SIZE = 1000;
    private static final int MAX_INFLIGHT = 100;
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withNumRetries(10)
            .withDelay(1_000)
            .withMaxDelay(60_000);

    private final ProjectsSupplier projectsSupplier;
    private final Executor executor;

    public AbstractAlertingFetcher(ProjectsSupplier projectsSupplier, Executor executor) {
        this.projectsSupplier = projectsSupplier;
        this.executor = executor;
    }

    @Override
    public CompletableFuture<Void> fetch(Consumer<SearchEvent> consumer) {
        return CompletableFutures.safeCall(() -> {
            var it = projectsSupplier.get().iterator();
            AsyncActorBody body = () -> {
                if (!it.hasNext()) {
                    return completedFuture(AsyncActorBody.DONE_MARKER);
                }

                var task = new Task(it.next(), consumer);
                task.next();
                return task.doneFuture;
            };

            AsyncActorRunner runner = new AsyncActorRunner(body, executor, MAX_INFLIGHT);
            return runner.start();
        });
    }

    protected abstract CompletableFuture<ListResult<T>> fetchNext(String projectId, String nextToken);

    protected abstract String folderId(T entry);

    protected abstract SearchEvent convert(T entry);

    private class Task {
        private final String projectId;
        private final CompletableFuture<Void> doneFuture = new CompletableFuture<>();
        private String nextToken = "";
        private final Consumer<SearchEvent> consumer;

        public Task(String projectId, Consumer<SearchEvent> consumer) {
            this.projectId = projectId;
            this.consumer = consumer;
        }

        private void next() {
            RetryCompletableFuture.runWithRetries(() -> fetchNext(projectId, nextToken), RETRY_CONFIG)
                    .whenComplete((response, e) -> {
                        if (e != null) {
                            doneFuture.completeExceptionally(e);
                            return;
                        }

                        try {
                            for (var entity : response.result) {
                                if ("".equals(folderId(entity))) {
                                    continue;
                                }

                                consumer.accept(convert(entity));
                            }

                            nextToken = response.nextToken;
                            if (response.result.size() < PAGE_SIZE) {
                                doneFuture.complete(null);
                            } else {
                                next();
                            }
                        } catch (Throwable e2) {
                            doneFuture.completeExceptionally(e2);
                        }
                    });
        }
    }

    static record ListResult<T>(String nextToken, List<T> result) {
    }
}
