package ru.yandex.solomon.gateway.cloud.search;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

import ru.yandex.cloud.resourcemanager.ResolvedFolder;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolver;
import ru.yandex.solomon.conf.db3.EntitiesDao;
import ru.yandex.solomon.conf.db3.ydb.Entity;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;

import static com.yandex.ydb.core.utils.Async.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class DashboardFetcher implements ResourceFetcher {
    private static final int BATCH_SIZE = 1000;
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withNumRetries(10)
            .withDelay(1_000)
            .withMaxDelay(60_000);

    private final EntitiesDao dao;
    private final CloudByFolderResolver cloudResolver;

    public DashboardFetcher(EntitiesDao dao, CloudByFolderResolver cloudResolver) {
        this.dao = dao;
        this.cloudResolver = cloudResolver;
    }

    @Override
    public CompletableFuture<Void> fetch(Consumer<SearchEvent> consumer) {
        return new Task(consumer).run();
    }

    private CompletableFuture<List<Entity>> retryListDashboards() {
        try {
            return RetryCompletableFuture.runWithRetries(dao::readAll, RETRY_CONFIG);
        } catch (Throwable e) {
            return failedFuture(e);
        }
    }

    private CompletableFuture<List<ResolvedFolder>> retryResolveFolders(Collection<String> folderIds) {
        try {
            return RetryCompletableFuture.runWithRetries(() -> cloudResolver.resolveClouds(folderIds), RETRY_CONFIG);
        } catch (Throwable e) {
            return failedFuture(e);
        }
    }

    private class Task {
        private final Consumer<SearchEvent> consumer;
        private final CompletableFuture<Void> doneFuture = new CompletableFuture<>();
        private final Map<String, String> cloudByFolderId = new HashMap<>();
        private final Set<String> unknownFolderIds = new HashSet<>();
        private Set<Entity> dashboards;

        public Task(Consumer<SearchEvent> consumer) {
            this.consumer = consumer;
        }

        public CompletableFuture<Void> run() {
            listDashboards();
            return doneFuture;
        }

        private void listDashboards() {
            retryListDashboards().thenAccept(result -> {
                dashboards = new HashSet<>(result);
                continueProcess();
            }).exceptionally(e -> {
                doneFuture.completeExceptionally(e);
                return null;
            });
        }

        private void resolveFolders() {
            retryResolveFolders(unknownFolderIds).thenAccept(resolved -> {
                for (var folder : resolved) {
                    cloudByFolderId.put(folder.getId(), folder.getCloudId());
                    unknownFolderIds.remove(folder.getId());
                }

                for (var folderId : unknownFolderIds) {
                    cloudByFolderId.put(folderId, "");
                }
                unknownFolderIds.clear();
                continueProcess();
            }).exceptionally(e -> {
                doneFuture.completeExceptionally(e);
                return null;
            });
        }

        private void continueProcess() {
            try {
                var it = dashboards.iterator();
                while (it.hasNext()) {
                    var dashboard = it.next();
                    var folderId = dashboard.getParentId();
                    if (unknownFolderIds.contains(folderId)) {
                        continue;
                    }

                    var cloudId = cloudByFolderId.get(folderId);
                    if (cloudId == null) {
                        unknownFolderIds.add(folderId);
                        if (unknownFolderIds.size() == BATCH_SIZE) {
                            resolveFolders();
                            return;
                        }
                        continue;
                    } else if (cloudId.isEmpty()) {
                        // cloud already removed
                        it.remove();
                        continue;
                    }

                    consumer.accept(SearchEvent.dashboard(cloudId, dashboard));
                    it.remove();
                }

                if (!unknownFolderIds.isEmpty()) {
                    resolveFolders();
                } else {
                    doneFuture.complete(null);
                }
            } catch (Throwable e) {
                doneFuture.completeExceptionally(e);
            }
        }
    }
}
