package ru.yandex.solomon.gateway.cloud.search;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.LockSubscriber;
import ru.yandex.solomon.locks.UnlockReason;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.time.DurationUtils;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class ReindexScheduler implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ReindexScheduler.class);

    private final DistributedLock lock;
    private final Duration period;
    private final long initialReindexDelayMillis;
    private final long minReindexPeriod;
    private final Consumer<SearchEvent> consumer;
    private final ExecutorService executor;
    private final ScheduledExecutorService timer;
    private final Clock clock;
    private final MetricRegistry registry;
    private final List<Target> targets;
    private volatile boolean closed;

    public ReindexScheduler(
            DistributedLock lock,
            Duration period,
            Duration initialReindexDelay,
            List<ResourceFetcher> fetchers,
            Consumer<SearchEvent> consumer,
            ExecutorService executor,
            ScheduledExecutorService timer,
            Clock clock,
            MetricRegistry registry)
    {
        this.lock = lock;
        this.period = period;
        this.minReindexPeriod = period.dividedBy(2).toMillis();
        this.initialReindexDelayMillis = initialReindexDelay.toMillis();
        this.consumer = consumer;
        this.executor = executor;
        this.timer = timer;
        this.clock = clock;
        this.registry = registry;
        this.targets = fetchers.stream()
                .map(Target::new)
                .collect(Collectors.toList());
        acquireLock();
    }

    private void acquireLock() {
        if (closed) {
            return;
        }

        lock.acquireLock(new LockSubscriber() {
            @Override
            public boolean isCanceled() {
                return closed;
            }

            @Override
            public void onLock(long seqNo) {
                logger.info("Acquire reindex lock, seqNo {}", seqNo);
                targets.forEach(Target::schedule);
            }

            @Override
            public void onUnlock(UnlockReason reason) {
                logger.info("Loose lock by reason: {}", reason);
                acquireLock();
            }
        }, 5, TimeUnit.MINUTES);
    }

    @Override
    public void close() {
        closed = true;
        targets.forEach(Target::close);
    }

    private class Target implements AutoCloseable {
        private final PingActorRunner actor;
        private final ResourceFetcher fetcher;
        private final AsyncMetrics metrics;
        private volatile long latestReindexTs;

        public Target(ResourceFetcher fetcher) {
            this.fetcher = fetcher;
            this.metrics = new AsyncMetrics(registry, "search.reindex.", Labels.of("target", fetcher.getClass().getSimpleName()));
            this.actor = PingActorRunner.newBuilder()
                    .executor(executor)
                    .timer(timer)
                    .operation("reindex search events from " + fetcher.getClass().getSimpleName())
                    .pingInterval(period)
                    .backoffDelay(Duration.ofMinutes(1))
                    .onPing(this::act)
                    .build();
        }

        public CompletableFuture<Void> act(int attempt) {
            if (closed) {
                return completedFuture(null);
            }

            if (!lock.isLockedByMe()) {
                return completedFuture(null);
            }

            long reindexTs = clock.millis();
            if (minReindexPeriod > reindexTs - latestReindexTs) {
                return completedFuture(null);
            }

            var future = fetcher.fetch(event -> {
                if (event.updatedAt >= reindexTs) {
                    return;
                }

                event.reindexAt = reindexTs;
                consumer.accept(event);
            }).thenAccept(ignore -> latestReindexTs = reindexTs);
            metrics.forFuture(future);
            return future;
        }

        public void schedule() {
            long delay = DurationUtils.randomize(initialReindexDelayMillis);
            timer.schedule(actor::forcePing, delay, TimeUnit.MILLISECONDS);
        }

        @Override
        public String toString() {
            return "Target{" +
                    "fetcher=" + fetcher +
                    ", latestReindexTs=" + Instant.ofEpochMilli(latestReindexTs) +
                    '}';
        }

        @Override
        public void close() {
            actor.close();
        }
    }
}
