package ru.yandex.solomon.gateway.cloud.search;

import java.time.Instant;
import java.util.Arrays;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;

import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
public class SearchEventConverter {
    private static final ObjectMapper mapper = new ObjectMapper();

    /**
     * Schema https://a.yandex-team.ru/arc/trunk/arcadia/cloud/search/schemas/v2/event.json
     */
    public static String toJson(SearchEvent event) {
        var root = mapper.createObjectNode();
        root.put("cloud_id", Nullables.orEmpty(event.cloudId));
        root.put("folder_id", Nullables.orEmpty(event.folderId));
        root.put("service", "monitoring");
        root.put("resource_id", Nullables.orEmpty(event.resourceId));
        root.put("resource_type", Nullables.orEmpty(event.resourceType));
        root.put("name", Nullables.orEmpty(event.name));
        root.putObject("attributes").put("description", Nullables.orEmpty(event.description));
        root.put("permission", Permission.CONFIGS_GET.getSlug());
        root.putArray("resource_path")
                .addAll(Arrays.asList(
                        resourcePath("resource-manager.cloud", Nullables.orEmpty(event.cloudId)),
                        resourcePath("resource-manager.folder", Nullables.orEmpty(event.folderId)),
                        resourcePath("monitoring." + Nullables.orEmpty(event.resourceType), Nullables.orEmpty(event.resourceId))
                ));

        if (event.deletedAt != 0) {
            String formatted = formatTime(event.deletedAt);
            root.put("timestamp", formatted);
            root.put("deleted", formatted);
        } else {
            root.put("timestamp", formatTime(event.updatedAt));
        }

        if (event.reindexAt != 0) {
            root.put("reindex_timestamp", formatTime(event.reindexAt));
        }

        try {
            return mapper.writeValueAsString(root);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    private static String formatTime(long millis) {
        return Instant.ofEpochMilli(millis).toString();
    }

    private static ObjectNode resourcePath(String type, String id) {
        ObjectNode node = mapper.createObjectNode();
        node.put("resource_type", type);
        node.put("resource_id", id);
        return node;
    }
}
