package ru.yandex.solomon.gateway.cloud.unified_agent;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

import com.google.common.io.Resources;
import org.apache.commons.lang3.text.StrSubstitutor;

import ru.yandex.solomon.config.gateway.UnifiedAgentInstallerConfig;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * @author Vladimir Gordiychuk
 */
public class UnifiedAgentInstaller {

    private final String userData;
    private final String installSh;

    public UnifiedAgentInstaller(UnifiedAgentInstallerConfig config) {
        checkArgument(!config.getApiAddress().isEmpty(), "api_address not specified");
        checkArgument(!config.getS3BucketAddress().isEmpty(), "s3_bucket_address not specified");
        this.userData = loadUserDataYml(config);
        this.installSh = loadInstallSh(config);
    }

    public byte[] userDataBytes() {
        return userData.getBytes(StandardCharsets.UTF_8);
    }

    public byte[] installShBytes() {
        return installSh.getBytes(StandardCharsets.UTF_8);
    }

    static String loadConfigYml(UnifiedAgentInstallerConfig config) {
        var params = toParams(config);
        return StrSubstitutor.replace(load("install/config.yml"), params);
    }

    static String loadUserDataYml(UnifiedAgentInstallerConfig config) {
        var params = toParams(config);
        return StrSubstitutor.replace(load("install/user_data.yml"), params);
    }

    static String loadSystemdUnit(UnifiedAgentInstallerConfig config) {
        var params = toParams(config);
        return StrSubstitutor.replace(load("install/systemd/unified_agent.service"), params);
    }

    static String loadInstallSh(UnifiedAgentInstallerConfig config) {
        var params = toParams(config);
        params.put("ua_config", loadConfigYml(config));
        params.put("ua_systemd_service", loadSystemdUnit(config));
        var template = load("install/install.sh");
        return StrSubstitutor.replace(template, params);
    }

    static Map<String, String> toParams(UnifiedAgentInstallerConfig config) {
        var result = new HashMap<String, String>();
        result.put("s3_bucket_address", config.getS3BucketAddress());
        result.put("api_address", config.getApiAddress());
        return result;
    }

    static String load(String name) {
        try {
            var url = Resources.getResource(UnifiedAgentInstaller.class, name);
            return Resources.toString(url, StandardCharsets.UTF_8);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
