package ru.yandex.solomon.gateway.cloud.billing;

import java.io.IOException;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagHolderStub;
import ru.yandex.solomon.gateway.cloud.billing.stockpile.StockpileUsage;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.util.time.Interval;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class LogBillingEndpointTest {

    private StringBuilder out;
    private MetricRegistry registry;
    private ShardByNumIdResolverStub resolver;
    private FeatureFlagHolderStub flags;
    private SessionStub ironSession;
    private SessionStub cloudSession;
    private LogBillingEndpoint endpoint;

    private Rate write;

    @Before
    public void setUp() {
        out = new StringBuilder();
        registry = new MetricRegistry();
        resolver = new ShardByNumIdResolverStub();
        flags = new FeatureFlagHolderStub();
        ironSession = new SessionStub();
        cloudSession = new SessionStub();
        endpoint = new LogBillingEndpoint(resolver, new BillingLogImpl(ironSession, cloudSession, flags, registry) {
            @Override
            protected void log(String json) {
                out.append(json).append('\n');
                super.log(json);
            }
        });

        write = registry.rate("billing.log.write");
    }

    @Test
    public void empty() {
        endpoint.push(new StockpileUsage(), interval(), 42).join();
        assertEquals(0, write.get());
        assertEquals("", out.toString());
    }

    @Test
    public void usage() {
        Shard shard = Shard.newBuilder()
            .setId("b1g86hg3icuf07_b1gfn2nk2sdf35v70o17_custom")
            .setNumId(ThreadLocalRandom.current().nextInt())
            .setProjectId("b1g86hg3icuf07")
            .setClusterName("b1gfn2nk2sdf35v70o17")
            .setClusterId("b1g86hg3icuf07_b1gfn2nk2sdf35v70o17")
            .setServiceName("custom")
            .setServiceId("b1g86hg3icuf07_custom")
            .build();

        resolver.addShard(shard);

        StockpileUsage usage = new StockpileUsage();
        usage.getUsage(MetricType.DGAUGE, shard.getNumId()).writeRecords = 10_000;
        usage.getUsage(MetricType.DGAUGE, shard.getNumId()).readRecords = 500;
        usage.getUsage(MetricType.DGAUGE, shard.getNumId()).storeRecords = 500_000;

        endpoint.push(usage, interval(), 43).join();
        assertEquals(2, write.get()); // write + read
        String[] lines = out.toString().lines().toArray(String[]::new);
        assertEquals(2, lines.length); // write + read
        assertWriteAndRead(shard, lines);

        //USE_UNIFIED_AGENT_FOR_BILLING(true) + USE_CLOUD_LOGBROKER_FOR_BILLING(false)
        assertEquals(0, ironSession.receivedMessages.size());
        flags.setFlag("b1g86hg3icuf07", FeatureFlag.USE_UNIFIED_AGENT_FOR_BILLING, true);
        endpoint.push(usage, interval(), 43).join();
        assertEquals(2, ironSession.receivedMessages.size());
        assertWriteAndRead(shard, ironSession.receivedMessages.toArray(String[]::new));

        //USE_UNIFIED_AGENT_FOR_BILLING(true) + USE_CLOUD_LOGBROKER_FOR_BILLING(true)
        assertEquals(0, cloudSession.receivedMessages.size());
        flags.setFlag("b1g86hg3icuf07", FeatureFlag.USE_CLOUD_LOGBROKER_FOR_BILLING, true);
        endpoint.push(usage, interval(), 43).join();
        assertEquals(2, cloudSession.receivedMessages.size());
        assertWriteAndRead(shard, cloudSession.receivedMessages.toArray(String[]::new));
    }

    private JsonNode parse(String line) {
        try {
            ObjectMapper mapper = new ObjectMapper();
            return mapper.readTree(line);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private Interval interval() {
        long now = System.currentTimeMillis();
        return Interval.millis(now - TimeUnit.MINUTES.toMillis(10), now);
    }

    private void assertWriteAndRead(Shard shard, String[] lines) {
        // writes
        {
            JsonNode target = parse(lines[0]);
            assertEquals(shard.getProjectId(), target.get("cloud_id").asText());
            assertEquals(shard.getClusterName(), target.get("folder_id").asText());
            assertEquals("monitoring.point.dgauge.write", target.get("schema").asText());
            assertEquals(10_000, target.get("usage").get("quantity").asInt());
        }
        // reads
        {
            JsonNode target = parse(lines[1]);
            assertEquals(shard.getProjectId(), target.get("cloud_id").asText());
            assertEquals(shard.getClusterName(), target.get("folder_id").asText());
            assertEquals("monitoring.point.dgauge.read", target.get("schema").asText());
            assertEquals(500, target.get("usage").get("quantity").asInt());
        }
        // TODO: uncomment when read and store will be reported to billing
//
//        {
//            JsonNode target = parse(lines[2]);
//            assertEquals(shard.getProjectId(), target.get("cloud_id").asText());
//            assertEquals(shard.getClusterName(), target.get("folder_id").asText());
//            assertEquals("monitoring.point.dgauge.store", target.get("schema").asText());
//            assertEquals(500_000, target.get("usage").get("quantity").asInt());
//        }
    }

}
