package ru.yandex.solomon.gateway.cloud.billing;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.gateway.cloud.billing.stockpile.StockpileUsage;
import ru.yandex.solomon.gateway.cloud.billing.stockpile.StockpileUsageProviderStub;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.LockDetail;
import ru.yandex.solomon.locks.LockService;
import ru.yandex.solomon.locks.LockServiceImpl;
import ru.yandex.solomon.locks.dao.memory.InMemoryLocksDao;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;
import ru.yandex.solomon.util.host.HostUtils;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileBillingResourceFetchManagerTest {

    private ManualClock clock;
    private ScheduledExecutorService timer;
    private LockService lockService;
    private BillingEndpointStub billingEndpoint;
    private volatile StockpileUsageProviderStub provider;
    private StockpileBillingResourceFetchManager manager;

    @Before
    public void setUp() {
        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(2, clock);
        var locksDao = new InMemoryLocksDao(clock);
        lockService = new LockServiceImpl(HostUtils.getFqdn(), locksDao, clock, timer, new MetricRegistry());
        billingEndpoint = new BillingEndpointStub();

        manager = new StockpileBillingResourceFetchManager(
            lockService,
            clock,
            timer,
            ForkJoinPool.commonPool(),
            TimeUnit.MINUTES.toMillis(10),
            billingEndpoint,
            () -> CompletableFuture.supplyAsync(() -> {
                provider = new StockpileUsageProviderStub();
                return provider;
            }),
            TimeUnit.DAYS.toMillis(1),
            new MetricRegistry());

        awaitBecomeLeader();
    }

    @After
    public void tearDown() {
        timer.shutdown();
        manager.close();
    }

    @Test
    public void empty() throws InterruptedException {
        awaitMessage();
        var request = billingEndpoint.getPushed();
        assertEquals(0, request.usage.usageByKey.size());
        assertTrue(TimeUnit.MINUTES.toMillis(10) <= request.interval.duration().toMillis());
    }

    @Test
    public void one() throws InterruptedException {
        int shardId = ThreadLocalRandom.current().nextInt();
        StockpileUsage.Usage expected = new StockpileUsage.Usage();
        expected.writeRecords = 100;
        expected.writeMetrics = 10;
        expected.readRecords = 1;
        expected.readMetrics = 1000;
        expected.storeRecords = 10_000;
        expected.storeMetrics = 55;

        provider.usage.getUsage(MetricType.DGAUGE, shardId).add(expected);

        awaitMessage();
        provider.usage = new StockpileUsage();

        var one = billingEndpoint.getPushed();
        assertEquals(1, one.usage.usageByKey.size());
        assertEquals(expected, one.usage.getUsage(MetricType.DGAUGE, shardId));
        assertTrue(TimeUnit.MINUTES.toMillis(10) <= one.interval.duration().toMillis());

        awaitMessage();
        var two = billingEndpoint.getPushed();
        assertNotEquals(one.interval, two.interval);
        assertEquals(0, two.usage.usageByKey.size());
    }

    @Test
    public void intervalNotOverlap() throws InterruptedException {
        awaitMessage();
        var one = billingEndpoint.getPushed();

        awaitMessage();
        var two = billingEndpoint.getPushed();

        awaitMessage();
        var tree = billingEndpoint.getPushed();

        assertEquals(tree.interval.getBeginMillis(), two.interval.getEndMillis());
        assertEquals(two.interval.getBeginMillis(), one.interval.getEndMillis());
        assertTrue(TimeUnit.MINUTES.toMillis(10) <= one.interval.duration().toMillis());
    }

    private void awaitMessage() throws InterruptedException {
        var sync = billingEndpoint.sync;
        while (!sync.await(1, TimeUnit.MILLISECONDS)) {
            clock.passedTime(30, TimeUnit.SECONDS);
        }
    }

    private void awaitBecomeLeader() {
        DistributedLock lock = lockService.distributedLock("BillingMaster");
        while (true) {
            Optional<LockDetail> optional = lock.getLockDetail(System.nanoTime()).join();
            if (optional.isEmpty()) {
                continue;
            }

            if (provider == null) {
                continue;
            }

            LockDetail owner = optional.get();
            if (HostUtils.getFqdn().equals(owner.owner())) {
                return;
            }
        }
    }
}
