package ru.yandex.solomon.gateway.cloud.billing.stockpile;

import java.util.List;
import java.util.concurrent.ThreadLocalRandom;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.model.protobuf.MetricType;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileCrossDcUsageProviderTest {

    private StockpileUsageProviderStub alice;
    private StockpileUsageProviderStub bob;

    private StockpileCrossDcUsageProvider provider;

    @Before
    public void setUp() {
        alice = new StockpileUsageProviderStub();
        bob = new StockpileUsageProviderStub();
        provider = new StockpileCrossDcUsageProvider(List.of(alice, bob));
    }

    @Test
    public void emptyResult() {
        var result = provider.getUsage();
        assertEquals(0, result.usageByKey.size());
    }

    @Test
    public void oneShardFromOneHost() {
        int shardOwnerId = ThreadLocalRandom.current().nextInt();
        var expected = alice.usage.getUsage(MetricType.DGAUGE, shardOwnerId);
        expected.writeMetrics = 1;
        expected.writeRecords = 100;
        expected.readMetrics = 5;
        expected.readRecords = 50;
        expected.storeMetrics = 25;
        expected.storeRecords = 10_000;

        var result = provider.getUsage();
        assertEquals(1, result.usageByKey.size());

        var actual = result.getUsage(MetricType.DGAUGE, shardOwnerId);
        assertEquals(expected, actual);
    }

    @Test
    public void oneShardFromTwoHost() {
        int shardOwnerId = ThreadLocalRandom.current().nextInt();
        var aliceUsage = alice.usage.getUsage(MetricType.DGAUGE, shardOwnerId);
        aliceUsage.writeMetrics = 1;
        aliceUsage.writeRecords = 100;
        aliceUsage.readMetrics = 5;
        aliceUsage.readRecords = 50;
        aliceUsage.storeMetrics = 25;
        aliceUsage.storeRecords = 10_000;

        var bobUsage = bob.usage.getUsage(MetricType.DGAUGE, shardOwnerId);
        bobUsage.writeMetrics = 2;
        bobUsage.writeRecords = 50;
        bobUsage.readMetrics = 10;
        bobUsage.readRecords = 1;
        bobUsage.storeMetrics = 55;
        bobUsage.storeRecords = 5_000;

        var expected = new StockpileUsage.Usage();
        expected.writeMetrics = 2;
        expected.writeRecords = 100;
        expected.readMetrics = 10;
        expected.readRecords = 50;
        expected.storeMetrics = 55;
        expected.storeRecords = 10_000;

        var result = provider.getUsage();
        assertEquals(1, result.usageByKey.size());

        var actual = result.getUsage(MetricType.DGAUGE, shardOwnerId);
        assertEquals(expected, actual);
    }

    @Test
    public void differentShards() {
        int aliceShard = ThreadLocalRandom.current().nextInt();
        int bobShard = ThreadLocalRandom.current().nextInt();

        var aliceUsage = alice.usage.getUsage(MetricType.DGAUGE, aliceShard);
        aliceUsage.writeMetrics = 1;
        aliceUsage.writeRecords = 100;
        aliceUsage.readMetrics = 5;
        aliceUsage.readRecords = 50;
        aliceUsage.storeMetrics = 25;
        aliceUsage.storeRecords = 10_000;

        var bobUsage = bob.usage.getUsage(MetricType.DGAUGE, bobShard);
        bobUsage.writeMetrics = 2;
        bobUsage.writeRecords = 50;
        bobUsage.readMetrics = 10;
        bobUsage.readRecords = 1;
        bobUsage.storeMetrics = 55;
        bobUsage.storeRecords = 5_000;

        var result = provider.getUsage();
        assertEquals(2, result.usageByKey.size());

        assertEquals(aliceUsage, result.getUsage(MetricType.DGAUGE, aliceShard));
        assertEquals(bobUsage, result.getUsage(MetricType.DGAUGE, bobShard));
    }
}
