package ru.yandex.solomon.gateway.cloud.billing.stockpile;

import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import com.google.common.net.HostAndPort;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.gateway.cloud.billing.BillingResourceFetcherStub;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;
import ru.yandex.solomon.util.host.HostUtils;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileHostTest {

    private ManualClock clock;
    private MetricRegistry registry;
    private ScheduledExecutorService timer;
    private BillingResourceFetcherStub fetcher;
    private StockpileHostUsageProvider host;

    @Before
    public void setUp() {
        clock = new ManualClock();
        registry = new MetricRegistry();
        timer = new ManualScheduledExecutorService(1, clock);
        fetcher = new BillingResourceFetcherStub(registry);
        host = new StockpileHostUsageProvider(HostAndPort.fromHost(HostUtils.getFqdn()), fetcher, value -> true, timer);
    }

    @After
    public void tearDown() {
        host.close();
        timer.shutdown();
    }

    @Test
    public void hostWithoutData() {
        var result = host.getUsage();
        assertEquals(0, result.usageByKey.size());
    }

    @Test
    public void rateMetricsReportsAsDiff() throws InterruptedException {
        awaitFetch(1);

        int ownerShardId = ThreadLocalRandom.current().nextInt();
        var type = MetricType.DGAUGE;
        setWrite(ownerShardId, type, 5, 50);
        setStore(ownerShardId, type, 10, 100);

        {
            awaitFetch(2);

            var result = host.getUsage();
            assertEquals(1, result.usageByKey.size());

            assertEquals(0, result.getUsage(type, ownerShardId).writeMetrics);
            assertEquals(0, result.getUsage(type, ownerShardId).writeRecords);
            assertEquals(100, result.getUsage(type, ownerShardId).storeRecords);
            assertEquals(10, result.getUsage(type, ownerShardId).storeMetrics);
        }

        {
            setWrite(ownerShardId, type, 6, 65);

            awaitFetch(2);
            var result = host.getUsage();
            assertEquals(1, result.usageByKey.size());

            assertEquals(1, result.getUsage(type, ownerShardId).writeMetrics);
            assertEquals(15, result.getUsage(type, ownerShardId).writeRecords);
            assertEquals(100, result.getUsage(type, ownerShardId).storeRecords);
            assertEquals(10, result.getUsage(type, ownerShardId).storeMetrics);
        }
    }

    @Test
    public void ignoreNegativeDelta() throws InterruptedException {
        awaitFetch(1);

        int ownerShardId = ThreadLocalRandom.current().nextInt();
        var type = MetricType.DGAUGE;
        setWrite(ownerShardId, type, 5, 50);
        setRead(ownerShardId, type, 1, 10);
        setStore(ownerShardId, type, 1000, 5000);

        awaitFetch(2);

        setWrite(ownerShardId, type, 50, 500);
        setRead(ownerShardId, type, 2, 12);

        awaitFetch(1);

        // process failed and start counter from zero

        setWrite(ownerShardId, type, 10, 20);
        setRead(ownerShardId, type, 0, 0);

        awaitFetch(1);

        setWrite(ownerShardId, type, 12, 22);
        setRead(ownerShardId, type, 1, 1);

        awaitFetch(2);

        {
            var result = host.getUsage();
            assertEquals(1, result.usageByKey.size());
            assertEquals(47, result.getUsage(type, ownerShardId).writeMetrics);
            assertEquals(452, result.getUsage(type, ownerShardId).writeRecords);
            assertEquals(2, result.getUsage(type, ownerShardId).readMetrics);
            assertEquals(3, result.getUsage(type, ownerShardId).readRecords);
            assertEquals(1000, result.getUsage(type, ownerShardId).storeMetrics);
            assertEquals(5000, result.getUsage(type, ownerShardId).storeRecords);
        }

        awaitFetch(2);
        {
            var result = host.getUsage();
            assertEquals(1, result.usageByKey.size());
            assertEquals(0, result.getUsage(type, ownerShardId).writeMetrics);
            assertEquals(0, result.getUsage(type, ownerShardId).writeRecords);
            assertEquals(0, result.getUsage(type, ownerShardId).readMetrics);
            assertEquals(0, result.getUsage(type, ownerShardId).readRecords);
            assertEquals(1000, result.getUsage(type, ownerShardId).storeMetrics);
            assertEquals(5000, result.getUsage(type, ownerShardId).storeRecords);
        }

        setWrite(ownerShardId, type, 13, 25);
        awaitFetch(2);
        {
            var result = host.getUsage();
            assertEquals(1, result.usageByKey.size());
            assertEquals(1, result.getUsage(type, ownerShardId).writeMetrics);
            assertEquals(3, result.getUsage(type, ownerShardId).writeRecords);
            assertEquals(0, result.getUsage(type, ownerShardId).readMetrics);
            assertEquals(0, result.getUsage(type, ownerShardId).readRecords);
            assertEquals(1000, result.getUsage(type, ownerShardId).storeMetrics);
            assertEquals(5000, result.getUsage(type, ownerShardId).storeRecords);
        }
    }

    private void awaitFetch(int count) throws InterruptedException {
        for (int index = 0; index < count; index++) {
            var sync = fetcher.sync;
            while (!sync.await(1, TimeUnit.MILLISECONDS)) {
                clock.passedTime(5, TimeUnit.SECONDS);
            }
        }
    }

    private void setWrite(int ownerShardId, MetricType type, long metrics, long records) {
        Labels labels = Labels.of("kind", type.name(), "projectId", "SOLOMON", "ownerId", Integer.toUnsignedString(ownerShardId));
        registry.rate("stockpile.write.sensors.rate", labels).set(metrics);
        registry.rate("stockpile.write.records.rate", labels).set(records);
    }

    private void setRead(int ownerShardId, MetricType type, long metrics, long records) {
        Labels labels = Labels.of("kind", type.name(), "projectId", "SOLOMON", "ownerId", Integer.toUnsignedString(ownerShardId));
        registry.rate("stockpile.read.sensors.rate", labels.add("producer", "USER")).set(metrics);
        registry.rate("stockpile.read.records.rate", labels.add("producer", "USER")).set(records);
    }

    private void setStore(int ownerShardId, MetricType type, long metrics, long records) {
        Labels labels = Labels.of("kind", type.name(), "projectId", "SOLOMON", "level", "total", "ownerId", Integer.toUnsignedString(ownerShardId));
        registry.gaugeInt64("stockpile.host.sensors.count", labels).set(metrics);
        registry.gaugeInt64("stockpile.host.record.count", labels).set(records);
    }
}
