package ru.yandex.solomon.gateway.cloud.billing.stockpile;

import java.util.concurrent.ThreadLocalRandom;
import java.util.function.IntPredicate;

import org.junit.Test;

import ru.yandex.monlib.metrics.JvmMemory;
import ru.yandex.monlib.metrics.JvmThreads;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.model.protobuf.MetricType;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileResourceUsageParserTest {

    @Test
    public void emptyParse() {
        var result = parse(new MetricRegistry());
        assertEquals(0, result.usageByKey.size());
    }

    @Test
    public void ignoreUnknownMetrics() {
        MetricRegistry registry = new MetricRegistry();
        JvmMemory.addMetrics(registry);
        JvmThreads.addMetrics(registry);

        StockpileUsage result = parse(registry);
        assertEquals(0, result.usageByKey.size());
    }

    @Test
    public void ignoreWritesOnNotCustomShards() {
        String ownerId = "1119893421";
        MetricRegistry registry = new MetricRegistry();
        {
            Labels labels = Labels.of("kind", "DGAUGE", "projectId", "SOLOMON", "ownerId", ownerId);
            registry.rate("stockpile.write.sensors.rate", labels).add(278788);
            registry.rate("stockpile.write.records.rate", labels).add(836364);
            registry.rate("stockpile.read.sensors.rate", labels.add("producer", "USER")).add(769);
            registry.rate("stockpile.read.records.rate", labels.add("producer", "USER")).add(10805);
        }
    }

    @Test
    public void parseOneOwner() {
        String ownerId = "1119893421";
        MetricRegistry registry = new MetricRegistry();
        {
            Labels labels = Labels.of("kind", "DGAUGE", "projectId", "SOLOMON", "ownerId", ownerId);
            registry.rate("stockpile.write.sensors.rate", labels).add(278788);
            registry.rate("stockpile.write.records.rate", labels).add(836364);
            registry.rate("stockpile.read.sensors.rate", labels.add("producer", "USER")).add(769);
            registry.rate("stockpile.read.records.rate", labels.add("producer", "USER")).add(10805);
        }

        var result = parse(registry, shardId -> false);
        assertEquals(1, result.usageByKey.size());

        var usage = result.getUsageOrNull(MetricType.DGAUGE, Integer.parseUnsignedInt(ownerId));
        assertNotNull(usage);

        assertEquals(0, usage.writeMetrics);
        assertEquals(0, usage.writeRecords);
        assertEquals(769, usage.readMetrics);
        assertEquals(10805, usage.readRecords);
        assertEquals(0, usage.storeRecords);
        assertEquals(0, usage.storeMetrics);
    }

    @Test
    public void parseMultiple() {
        StockpileUsage usage = new StockpileUsage();
        int ownerIdOne = ThreadLocalRandom.current().nextInt();
        int ownerIdTwo = ThreadLocalRandom.current().nextInt();

        {
            var gauge = usage.getUsage(MetricType.DGAUGE, ownerIdOne);
            gauge.writeRecords = 100;
            gauge.writeMetrics = 10;
            gauge.readRecords = 0;
            gauge.readMetrics = 0;
            gauge.storeRecords = 1000;
            gauge.storeMetrics = 10;
        }

        {
            var rate = usage.getUsage(MetricType.RATE, ownerIdTwo);
            rate.writeRecords = 14200;
            rate.writeMetrics = 241;
            rate.readMetrics = 10;
            rate.readRecords = 1123;
            rate.storeMetrics = 1100;
            rate.storeRecords = 11245111;
        }

        var registry = new MetricRegistry();
        add(registry, usage);

        var parsed = parse(registry);
        assertEquals(2, parsed.usageByKey.size());
        assertNotNull(parsed.getUsageOrNull(MetricType.DGAUGE, ownerIdOne));
        assertNotNull(parsed.getUsageOrNull(MetricType.RATE, ownerIdTwo));
        assertUsageEqual(usage.getUsage(MetricType.DGAUGE, ownerIdOne), parsed.getUsage(MetricType.DGAUGE, ownerIdOne));
        assertUsageEqual(usage.getUsage(MetricType.RATE, ownerIdTwo), parsed.getUsage(MetricType.RATE, ownerIdTwo));
    }

    private static void assertUsageEqual(StockpileUsage.Usage expectUsage, StockpileUsage.Usage actualUsage) {
        assertEquals(expectUsage.writeMetrics, actualUsage.writeMetrics);
        assertEquals(expectUsage.writeRecords, actualUsage.writeRecords);
        assertEquals(expectUsage.readMetrics, actualUsage.readMetrics);
        assertEquals(expectUsage.readRecords, actualUsage.readRecords);
        assertEquals(expectUsage.storeMetrics, actualUsage.storeMetrics);
        assertEquals(expectUsage.storeRecords, actualUsage.storeRecords);
    }

    private void add(MetricRegistry registry, StockpileUsage usage) {
        for (var entry : usage.usageByKey.long2ObjectEntrySet()) {
            int ownerId = StockpileUsage.ownerShardId(entry.getLongKey());
            MetricType type = StockpileUsage.type(entry.getLongKey());
            add(registry, Integer.toUnsignedString(ownerId), type.name(), entry.getValue());
            add(registry, Integer.toUnsignedString(ownerId), "total", entry.getValue());
            add(registry, "total", type.name(), entry.getValue());
        }
    }

    private void add(MetricRegistry registry, String ownerId, String kind, StockpileUsage.Usage usage) {
        {
            Labels labels = Labels.of("kind", kind, "projectId", "SOLOMON", "ownerId", ownerId);
            registry.rate("stockpile.write.sensors.rate", labels.add("producer", "UNKNOWN_PRODUCER")).add(usage.writeMetrics);
            registry.rate("stockpile.write.records.rate", labels.add("producer", "UNKNOWN_PRODUCER")).add(usage.writeRecords);
            registry.rate("stockpile.read.sensors.rate", labels.add("producer", "USER")).add(usage.readMetrics);
            registry.rate("stockpile.read.records.rate", labels.add("producer", "USER")).add(usage.readRecords);
        }
        {
            Labels labels = Labels.of("kind", kind, "projectId", "SOLOMON", "level", "total", "ownerId", ownerId);
            registry.gaugeInt64("stockpile.host.record.count", labels).add(usage.storeRecords);
            registry.gaugeInt64("stockpile.host.sensors.count", labels).add(usage.storeMetrics);
        }
    }

    private StockpileUsage parse(MetricSupplier supplier) {
        return parse(supplier, shardId -> true);
    }

    private StockpileUsage parse(MetricSupplier supplier, IntPredicate filter) {
        StockpileResourceUsageParser parser = new StockpileResourceUsageParser(filter);
        supplier.supply(0, parser);
        return parser.getUsage();
    }
}
