package ru.yandex.solomon.gateway.cloud.search;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.collect.Sets;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.client.stub.AlertApiStub;
import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.alert.protobuf.TCreateAlertRequest;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class AlertFetcherTest {

    private AlertApiStub alertApi;
    private AlertFetcher fetcher;
    private Set<String> projects;

    @Before
    public void setUp() {
        alertApi = new AlertApiStub();
        projects = Sets.newConcurrentHashSet();
        fetcher = new AlertFetcher(() -> Set.copyOf(projects), alertApi, ForkJoinPool.commonPool());
    }

    @Test
    public void reindexEmpty() {
        assertEquals(List.of(), consume());
    }

    @Test
    public void reindexShardsWithoutAlerts() {
        for (int index = 0; index < 10_000; index++) {
            projects.add("p" + index);
        }

        assertEquals(List.of(), consume());
    }

    @Test
    public void reindexAlerts() {
        var now = System.currentTimeMillis();
        List<TAlert> alerts = List.of(
                TAlert.newBuilder()
                        .setProjectId("p1")
                        .setId("a1")
                        .setName("alice")
                        .setFolderId("f1")
                        .setCreatedAt(now)
                        .setUpdatedAt(now)
                        .build(),

                TAlert.newBuilder()
                        .setProjectId("p2")
                        .setId("a2")
                        .setName("bob")
                        .setFolderId("f2")
                        .setCreatedAt(now)
                        .setUpdatedAt(now)
                        .build());

        projects.add("p1");
        projects.add("p2");
        for (var alert : alerts) {
            alertApi.createAlert(TCreateAlertRequest.newBuilder()
                    .setAlert(alert)
                    .build()).join();
        }

        var actual = consume();
        var expected = alerts.stream()
                .map(SearchEvent::alert)
                .sorted(Comparator.comparing(o -> o.resourceId))
                .collect(Collectors.toList());
        actual.sort(Comparator.comparing(o -> o.resourceId));
        assertEquals(expected, actual);
    }

    @Test
    public void manyAlertsInProject() {
        var now = System.currentTimeMillis();
        int size = 1000 + ThreadLocalRandom.current().nextInt(2000);
        List<TAlert> alerts = IntStream.range(0, size)
                .mapToObj(idx -> {
                    return TAlert.newBuilder()
                            .setProjectId("prj")
                            .setId("id"+idx)
                            .setName("name"+idx)
                            .setFolderId("folder")
                            .setCreatedAt(now)
                            .setUpdatedAt(now)
                            .build();
                })
                .collect(Collectors.toList());

        projects.add("prj");
        alerts.parallelStream()
                .map(alert -> {
                    return alertApi.createAlert(TCreateAlertRequest.newBuilder()
                            .setAlert(alert)
                            .build());
                })
                .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOfVoid))
                .join();

        var actual = consume();
        var expected = alerts.stream()
                .map(SearchEvent::alert)
                .sorted(Comparator.comparing(o -> o.resourceId))
                .collect(Collectors.toList());
        actual.sort(Comparator.comparing(o -> o.resourceId));
        assertEquals(expected, actual);
    }

    private List<SearchEvent> consume() {
        var queue = new ConcurrentLinkedQueue<SearchEvent>();
        fetcher.fetch(queue::add).join();
        return new ArrayList<>(queue);
    }
}
