package ru.yandex.solomon.gateway.cloud.search;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ForkJoinPool;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.alert.client.stub.NotificationApiStub;
import ru.yandex.solomon.alert.protobuf.TCreateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.notification.TNotification;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class ChannelFetcherTest {

    private NotificationApiStub channelApi;
    private ResourceFetcher fetcher;
    private Set<String> projects;

    @Before
    public void setUp() {
        channelApi = new NotificationApiStub();
        projects = new HashSet<>();
        fetcher = new ChannelFetcher(() -> Set.copyOf(projects), channelApi, ForkJoinPool.commonPool());
    }

    @Test
    public void reindexEmpty() {
        assertEquals(List.of(), consume());
    }

    @Test
    public void reindexShardsWithoutAlerts() {
        for (int index = 0; index < 100; index++) {
            projects.add("p" + index);
        }

        assertEquals(List.of(), consume());
    }

    @Test
    public void reindexChannels() {
        var now = System.currentTimeMillis();
        List<TNotification> channels = List.of(
                TNotification.newBuilder()
                        .setProjectId("p1")
                        .setId("a1")
                        .setName("alice")
                        .setFolderId("f1")
                        .setCreatedAt(now)
                        .setUpdatedAt(now)
                        .build(),

                TNotification.newBuilder()
                        .setProjectId("p2")
                        .setId("a2")
                        .setName("bob")
                        .setFolderId("f2")
                        .setCreatedAt(now)
                        .setUpdatedAt(now)
                        .build());

        projects.add("p1");
        projects.add("p2");
        for (var channel : channels) {
            channelApi.createNotification(TCreateNotificationRequest.newBuilder()
                    .setNotification(channel)
                    .build()).join();
        }

        var actual = consume();
        var expected = channels.stream()
                .map(SearchEvent::channel)
                .sorted(Comparator.comparing(o -> o.resourceId))
                .collect(Collectors.toList());
        actual.sort(Comparator.comparing(o -> o.resourceId));
        assertEquals(expected, actual);
    }

    private List<SearchEvent> consume() {
        var queue = new ConcurrentLinkedQueue<SearchEvent>();
        fetcher.fetch(queue::add).join();
        return new ArrayList<>(queue);
    }
}
