package ru.yandex.solomon.gateway.cloud.search;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolverStub;
import ru.yandex.solomon.conf.db3.memory.InMemoryEntitiesDao;
import ru.yandex.solomon.conf.db3.ydb.Entity;
import ru.yandex.solomon.core.container.ContainerType;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class DashboardFetcherTest {

    private InMemoryEntitiesDao dao;
    private CloudByFolderResolverStub cloudResolver;
    private DashboardFetcher fetcher;

    @Before
    public void setUp() {
        dao = new InMemoryEntitiesDao();
        cloudResolver = new CloudByFolderResolverStub();
        fetcher = new DashboardFetcher(dao, cloudResolver);
    }

    @Test
    public void reindexEmpty() {
        var result = consume();
        assertEquals(List.of(), result);
    }

    @Test
    public void reindexIgnoreWithoutFolder() {
        Entity dashboard = Entity.newBuilder()
                .setId("id")
                .setName("name")
                .setParentId("not_exist_folder_id")
                .setContainerType(ContainerType.FOLDER)
                .build();
        dao.add(dashboard);
        assertEquals(List.of(), consume());
    }

    @Test
    public void reindexDashboards() {
        cloudResolver.add("folder-2", "project");
        cloudResolver.add("folder-1", "project");
        List<Entity> dashboards = List.of(
                Entity.newBuilder()
                        .setId("one")
                        .setName("one-name")
                        .setParentId("folder-2")
                        .setContainerType(ContainerType.FOLDER)
                        .build(),

                Entity.newBuilder()
                        .setId("two")
                        .setName("two-name")
                        .setParentId("folder-2")
                        .setContainerType(ContainerType.FOLDER)
                        .build(),

                Entity.newBuilder()
                        .setId("three")
                        .setName("three-name")
                        .setParentId("folder-1")
                        .setContainerType(ContainerType.FOLDER)
                        .build()
        );

        dao.addAll(dashboards);

        var actual = consume();
        var expected = dashboards.stream()
                .map(entity -> SearchEvent.dashboard("project", entity))
                .sorted(Comparator.comparing(o -> o.resourceId))
                .collect(Collectors.toList());
        actual.sort(Comparator.comparing(o -> o.resourceId));
        assertEquals(expected, actual);
    }

    @Test
    public void reindexMany() {
        var random = ThreadLocalRandom.current();
        List<SearchEvent> expected = new ArrayList<>(100_000);
        for (int index = 0; index < 10_000; index++) {
            var folderId = "folderId-" + random.nextInt(1_500);
            var cloudId = cloudResolver.get(folderId);
            if (cloudId == null) {
                cloudId = "cloudId-" + random.nextInt(100);
                cloudResolver.add(folderId, cloudId);
            }
            var now = System.currentTimeMillis();
            Entity dashboard = Entity.newBuilder()
                    .setId("dashboardId-" + ThreadLocalRandom.current().nextLong())
                    .setName("dashboardName-" + ThreadLocalRandom.current().nextInt(10))
                    .setParentId(folderId)
                    .setContainerType(ContainerType.FOLDER)
                    .setUpdatedAt(now)
                    .build();
            dao.add(dashboard);
            expected.add(SearchEvent.dashboard(cloudId, dashboard));
        }

        var actual = consume();
        expected.sort(Comparator.comparing(o -> o.resourceId));
        actual.sort(Comparator.comparing(o -> o.resourceId));
        assertArrayEquals(expected.toArray(), actual.toArray());
    }

    private List<SearchEvent> consume() {
        var queue = new ConcurrentLinkedQueue<SearchEvent>();
        fetcher.fetch(queue::add).join();
        return new ArrayList<>(queue);
    }
}
