package ru.yandex.solomon.gateway.cloud.search;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.locks.DistributedLockStub;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;
import ru.yandex.solomon.util.host.HostUtils;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class ReindexSchedulerTest {

    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(1, TimeUnit.MINUTES)
            .build();

    private ResourceFetcherStub fetcherOne;
    private ResourceFetcherStub fetcherTwo;
    private ManualClock clock;
    private ManualScheduledExecutorService timer;
    private DistributedLockStub lock;
    private ReindexScheduler scheduler;
    private ArrayBlockingQueue<SearchEvent> events;

    @Before
    public void setUp() throws Exception {
        fetcherOne = new ResourceFetcherStub();
        fetcherTwo = new ResourceFetcherStub();

        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(1, clock);
        lock = new DistributedLockStub(clock);
        events = new ArrayBlockingQueue<>(1000);
        scheduler = new ReindexScheduler(
                lock,
                Duration.ofMinutes(10),
                Duration.ZERO,
                List.of(fetcherOne, fetcherTwo),
                event -> {
                    events.offer(event);
                },
                ForkJoinPool.commonPool(),
                timer,
                clock,
                new MetricRegistry());
    }

    @After
    public void tearDown() throws Exception {
        scheduler.close();
        timer.shutdownNow();
    }

    @Test
    public void reindexWhenBecomeLeader() throws InterruptedException {
        var expect = randomEvent();

        fetcherOne.add(expect);
        lock.setOwner(HostUtils.getFqdn());
        expectEvent(expect);
    }

    @Test
    public void reindexPereodically() throws InterruptedException {
        var expected = randomEvent();
        fetcherOne.add(expected);
        lock.setOwner(HostUtils.getFqdn());

        expectEvent(expected);
        for (int index = 0; index < 3; index++) {
            expectEvent(expected);
        }
    }

    @Test
    public void retryFailed() throws InterruptedException {
        var one = randomEvent();
        var two = randomEvent();

        fetcherOne.add(one);
        fetcherTwo.add(two);
        fetcherTwo.error = new RuntimeException("unavailable 1");

        lock.setOwner(HostUtils.getFqdn());

        expectEvent(one);

        fetcherOne.error = new RuntimeException("unavailable 2");
        fetcherTwo.error = null;
        expectEvent(two);

        fetcherTwo.error = new RuntimeException("unavailable 3");
        fetcherOne.error = null;
        expectEvent(one);
    }

    private void expectEvent(SearchEvent expected) throws InterruptedException {
        SearchEvent event;
        while ((event = events.poll(1, TimeUnit.MILLISECONDS)) == null) {
            clock.passedTime(1, TimeUnit.MINUTES);
        }
        assertNotEquals(0, event.reindexAt);
        event.reindexAt = 0;
        expected.reindexAt = 0;
        assertEquals(expected, event);
    }

    private SearchEvent randomEvent() {
        return new SearchEvent().setCloudId("cloud")
                .setFolderId("folder")
                .setResourceId("id-" + ThreadLocalRandom.current().nextLong())
                .setName("name")
                .setUpdatedAt(System.currentTimeMillis() - ThreadLocalRandom.current().nextLong(1_000));
    }
}
