#include "http.h"

#include <solomon/services/ingestor/lib/api/service.h>
#include <solomon/services/ingestor/lib/config/config.h>
#include <solomon/services/ingestor/lib/requests_tracer/request_tracer.h>
#include <solomon/services/ingestor/lib/shard_manager/shard_manager.h>

#include <solomon/services/dataproxy/lib/memstore/watcher.h>

#include <solomon/libs/cpp/actors/metrics/actor_runtime_metrics.h>
#include <solomon/libs/cpp/actors/runtime/actor_runtime.h>
#include <solomon/libs/cpp/clients/memstore/rpc.h>
#include <solomon/libs/cpp/cluster_map/cluster.h>
#include <solomon/libs/cpp/cluster_membership/actor.h>
#include <solomon/libs/cpp/cluster_membership/events.h>
#include <solomon/libs/cpp/cluster_membership/grpc_service.h>
#include <solomon/libs/cpp/conf_db/db.h>
#include <solomon/libs/cpp/config_includes/config_includes.h>
#include <solomon/libs/cpp/grpc/executor/limits.h>
#include <solomon/libs/cpp/grpc/server/server.h>
#include <solomon/libs/cpp/host_resolver/host_resolver.h>
#include <solomon/libs/cpp/load_info_service/load_info_service.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/metering/metric_suppliers.h>
#include <solomon/libs/cpp/minidump/minidump.h>
#include <solomon/libs/cpp/secrets/secrets.h>
#include <solomon/libs/cpp/selfmon/service/proto_page.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/signals/signals.h>
#include <solomon/libs/cpp/clients/slicer/slicelet_listener.h>
#include <solomon/libs/cpp/clients/slicer/slicer_client.h>
#include <solomon/libs/cpp/threading/metrics/thread_pool_metrics.h>
#include <solomon/libs/cpp/threading/pool/pool.h>
#include <solomon/libs/cpp/ydb/driver.h>
#include <solomon/protos/configs/rpc/rpc_config.pb.h>

#include <library/cpp/actors/core/process_stats.h>
#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/protobuf/util/pb_io.h>

#include <util/folder/path.h>
#include <util/generic/size_literals.h>
#include <util/generic/xrange.h>
#include <util/system/getpid.h>
#include <util/system/hostname.h>

using namespace NActors;
using namespace NMonitoring;
using namespace NSolomon::NDb;
using namespace NSolomon::NIngestor;
using namespace NSolomon::NLoadInfo;
using namespace NSolomon::NSlicerClient;
using namespace NSolomon;
using namespace yandex::monitoring::ingestor;
using yandex::solomon::config::rpc::TGrpcServerConfig;
using yandex::solomon::config::http::HttpServerConfig;

using IMetricSupplierPtr = TAtomicSharedPtr<NMonitoring::IMetricSupplier>;

std::shared_ptr<IThreadPool> CreateParsingThreadPool(const DataProcessorsExecutorConfig& config, NMonitoring::TMetricRegistry& registry) {
    TString name = "processing";

    IThreadPool::TParams params;
    params.SetBlocking(false);
    params.SetCatching(true);
    params.SetThreadName(name);

    auto metrics = std::make_shared<TThreadPoolMetrics>(name, registry);
    return CreateThreadPool(config.threads_num(), 0, std::move(params), std::move(metrics));
}

void ResolveAllAddresses(IngestorConfig& conf) {
    auto replace = [r = CachingResolver(ConductorGroupResolver())](auto* addresses) {
        TAddressSet resolved;
        for (const TString& a: *addresses) {
            Cerr << "resolving " << a << Endl;
            r->Resolve(a, &resolved);
        }
        addresses->Clear();
        for (const TString& a: resolved) {
            addresses->Add()->assign(a);
        }
    };

    replace(conf.mutable_coremon_client()->mutable_addresses());
    replace(conf.mutable_memstore_client()->mutable_addresses());
}

TString GetYasmPrefix(const IngestorConfig& config) {
    TStringBuf prefix;
    if (config.HasYasmPrefix()) {
        auto& proto = config.GetYasmPrefix();
        prefix = TStringBuf(proto.GetPrefix().data(), proto.GetPrefix().size());
    } else {
        prefix = TStringBuf("yasm_pull_");
    }

    Y_ENSURE(
            prefix.StartsWith(TStringBuf("yasm_")) && prefix.back() == '_',
            "invalid yasm prefix:" << prefix << ", yasm_prefix must starts with \"yasm_\" and ends with \"_\"");

    Cerr << "yasm prefix:" << prefix << '\n';

    return TString{prefix};
}

int RunIngestor(IngestorConfig& config, NSecrets::ISecretProvider& secretProvider) {
    NSolomon::NGrpc::SetThreadsLimit(2);
    Cerr << "Starting Ingestor (pid: " << GetPID() << ", SVN revision: " << GetArcadiaLastChange() << ')' << Endl;

    InitSignals();

    auto ingestorCluster = MakeClusterMapFromConfig(config);
    Cerr << "Cluster resolved into: " << *ingestorCluster << Endl;

    auto metrics = std::make_shared<TMetricRegistry>();
    auto actorRuntime = TActorRuntime::Create(config.actor_system(), metrics);
    actorRuntime->Start();

    auto shardMeteringMetricsRepositoryPtr = std::make_shared<TShardMeteringMetricsRepository>();
    metrics->IntGauge({{"sensor", "version"}, {"revision", GetArcadiaLastChange()}})->Set(1);
    auto shardMetricsSupplier = CreateShardMetricsSupplier(shardMeteringMetricsRepositoryPtr);
    auto hostMetricsSupplier = CreateHostMetricsSupplier(shardMeteringMetricsRepositoryPtr);

    std::unique_ptr<NSolomon::NHttp::THttpServer> httpServer;
    if (config.has_http_api_server()) {
        httpServer = NSolomon::NIngestor::StartHttpServer(
            *actorRuntime,
            *shardMetricsSupplier,
            *hostMetricsSupplier,
            metrics,
            config.http_api_server());

        NSelfMon::RegisterPage(
            actorRuntime->ActorSystem(),
            "/config", "Config",
            NSelfMon::ProtoPage(std::make_shared<IngestorConfig>(config)));
    }

    ui32 apiExecutorPool = actorRuntime->FindExecutorByName(config.api_server().GetThreadPoolName());
    ui32 shardExecutorPool = actorRuntime->FindExecutorByName("shard");

    // YDB Connection
    auto connection = NDb::CreateYdbConnection(NDb::CreateDriver(config.ydb_config(), secretProvider), *metrics);
    auto pathPrefix = config.config_path_prefix();
    Y_ENSURE(!pathPrefix.empty() && pathPrefix.back() == '/', "Wrong path prefix format");
    auto shardDao = connection->CreateShardDao(TShardTables{
        .ShardTablePath = pathPrefix + "Shard",
        .NumPcsTablePath = pathPrefix + "ShardPcsKey",
        .NumIdPcsTablePath = pathPrefix + "ShardPcsNumId",
    });
    auto serviceDao = connection->CreateServiceDao(pathPrefix + "Service");
    auto projectDao = connection->CreateProjectDao(pathPrefix + "Project");
    auto clusterDao = connection->CreateClusterDao(pathPrefix + "Cluster");

    auto dataProcessorsExecutor = CreateParsingThreadPool(config.processors_executor_config(), *metrics);

    std::shared_ptr<IClusterRpc<NMemStore::IMemStoreRpc>> memStoreRpc;
    TActorId memStoreClusterWatcher;
    if (config.has_memstore_client()) {
        memStoreRpc = NMemStore::CreateClusterGrpc(config.memstore_client(), *metrics, config.client_id());
        memStoreClusterWatcher = actorRuntime->Register(NDataProxy::MemStoreClusterWatcher(
            memStoreRpc,
            memStoreRpc->Addresses(),
            TDuration::Seconds(5)).release());
    }

    IRequestTracerPtr tracer = CreateRequestTracer(config.request_trace_config(), *actorRuntime, *metrics);

    auto loadBalancingMode = config.load_balancing_mode();
    bool isStandalone = loadBalancingMode == IngestorConfig::STANDALONE;

    TActorId sliceletId;
    switch (loadBalancingMode) {
        case IngestorConfig::STANDALONE:
            break;
        case IngestorConfig::USE_SLICER: {
            TStringBuilder localAddress;
            localAddress << FQDNHostName() << ':' << config.api_server().GetPort(0);

            sliceletId = actorRuntime->Register(
                CreateSliceletListenerActor(ConstructSliceletListenerOptions(
                    config.slicelet_config(),
                    *metrics,
                    localAddress)));
            break;
        }
        default:
            ythrow yexception() << "unknown load_balancing_mode: "
                                << IngestorConfig::LoadBalancingMode_Name(loadBalancingMode);
    }

    TActorId shardManagerId = actorRuntime->Register(
            CreateShardManager(
                    // TODO(ivanzhukov): pass a config via one global context
                    config,
                    shardExecutorPool,
                    shardDao,
                    clusterDao,
                    serviceDao,
                    projectDao,
                    *metrics,
                    shardMeteringMetricsRepositoryPtr,
                    dataProcessorsExecutor.get(),
                    // TODO(msherbakov): this probably will not be required as soon as we have a real load balancer
                    *ingestorCluster,
                    // indicates that shard manager has to make up assignments by itself
                    isStandalone,
                    memStoreRpc,
                    memStoreClusterWatcher,
                    tracer,
                    sliceletId,
                    GetYasmPrefix(config)
            )
    );

    TActorId clusterMembershipActorId;
    if (config.has_cluster_membership_config()) {
        auto clusterMembershipActor = NClusterMembership::CreateClusterMembershipActor(
                "ingestor",
                config.cluster_membership_config(),
                config.api_server().GetPort(0),
                *metrics,
                {},
                config.client_id(),
                NClusterMembership::EMembershipMode::Passive);
        clusterMembershipActorId = actorRuntime->Register(clusterMembershipActor.Release());
    }

    AddActorRuntimeMetrics(*actorRuntime, *metrics);

    auto grpcSrv = NSolomon::NGrpc::MakeGRpcServerFromConfig(actorRuntime->ActorSystem(), *metrics, config.api_server());

    grpcSrv->AddService(NApi::CreateIngestorService(actorRuntime->ActorSystem(), *metrics, apiExecutorPool, shardManagerId));
    grpcSrv->AddService(CreateLoadInfoService(actorRuntime->ActorSystem(), *metrics, apiExecutorPool, shardManagerId));

    if (config.has_cluster_membership_config()) {
        MON_DEBUG_C(actorRuntime->ActorSystem(), Ingestor, "setting the ClusterMembership service up");

        grpcSrv->AddService(NClusterMembership::CreateClusterMembershipService(
                actorRuntime->ActorSystem(),
                *metrics,
                apiExecutorPool,
                clusterMembershipActorId));
    }

    grpcSrv->Start();

    sigset_t blockMask;
    SigEmptySet(&blockMask);
    while (true) {
        SigSuspend(&blockMask);

        if (NeedTerminate) {
            actorRuntime->EmergencyLog("Ingestor was terminated");
            break;
        } else if (NeedReopenLog) {
            NeedReopenLog = 0;
            actorRuntime->ReopenLog();
            TLogBackend::ReopenAllBackends();
        }
    }

    // everything must be stopped before the actor system!
    grpcSrv->Stop();
    if (httpServer) {
        httpServer.reset();
    }

    // destroy here all core actors
    auto future = actorRuntime->AsyncPoison(std::set{clusterMembershipActorId}, TDuration::Seconds(5));
    future.GetValueSync();

    actorRuntime->Stop();
    return 0;
}

int main(int argc, const char** argv) {
    NLastGetopt::TOpts opts;
    opts.AddLongOption("config", "path to configuration file")
        .Required()
        .RequiredArgument("<path>")
        .Completer(NLastGetopt::NComp::File("*.conf"));
    opts.AddLongOption("secrets", "path to secerets file")
        .Required()
        .RequiredArgument("<path>")
        .Completer(NLastGetopt::NComp::File("*.secrets"));
    opts.AddVersionOption();
    opts.AddHelpOption();
    opts.AddCompletionOption("ingestor");
    opts.SetFreeArgsNum(0);

    try {
        NLastGetopt::TOptsParseResult r(&opts, argc, argv);
        Cerr << "Starting Ingestor (pid: " << GetPID() << ", SVN revision: " << GetArcadiaLastChange() << ')' << Endl;

        auto config = ParseTextProto(r.Get("config"));
        MergeIncludes(config);
        ResolveAllAddresses(config);

        SetupMinidump(config.minidump_config());

        TFsPath secrets(r.Get("secrets"));
        auto secretProvider = secrets.Exists()
            ? NSecrets::FileSecretProvider(secrets.GetPath())
            : NSecrets::EmptySecretProvider();

        return RunIngestor(config, *secretProvider);
    } catch (const NLastGetopt::TUsageException&) {
        opts.PrintUsage("ingestor");
    } catch (...) {
        Cerr << "Unhandled exception: " << CurrentExceptionMessage() << ". Process will terminate" << Endl;
    }

    return 1;
}
