#include <solomon/services/ingestor/api/ingestor_grpc_service.grpc.pb.h>
#include <solomon/services/ingestor/lib/shard_manager/shard_manager.h>

#include <solomon/libs/cpp/grpc/server/handler.h>
#include <solomon/libs/cpp/proto_convert/format.h>
#include <solomon/libs/cpp/proto_convert/labels.h>

using namespace NActors;

using yandex::monitoring::ingestor::TCreateShardRequest;
using yandex::monitoring::ingestor::TCreateShardResponse;

namespace NSolomon::NIngestor::NApi {
namespace {

class TCreateShardHandler: public NGrpc::TBaseHandler<TCreateShardHandler> {
public:
    TCreateShardHandler(TActorId shardManager) noexcept
        : ShardManager_(shardManager)
    {
    }

public:
    STATEFN(HandleEvent) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TShardManagerEvents::TEvCreateShardResult, OnCreateShardResult);
        }
    }

    EMode HandleRequest(NGrpc::TRequestState* req) {
        auto* msg = req->GetMessage<TCreateShardRequest>();
        auto* ev = new TShardManagerEvents::TEvCreateShard{
            SelfId(),
            msg->GetProjectId(),
            msg->GetClusterName(),
            msg->GetServiceName()
        };
        Send(ShardManager_, ev, 0, req->ToCookie());

        // cleanup request's data from the arena ASAP
        req->Arena()->Reset();
        return EMode::Async;
    }

    void OnCreateShardResult(const TShardManagerEvents::TEvCreateShardResult::TPtr& ev) {
        auto req = NGrpc::TRequestState::FromCookie(ev->Cookie);
        if (req->IsReplied()) {
            // request was already expired
            return;
        }

        auto& result = ev->Get()->Result;
        if (result.Fail()) {
            // TODO: set proper error type
            req->SendError(grpc::StatusCode::UNKNOWN, result.Error().MessageString());
            return;
        }

        try {
            auto* resp = google::protobuf::Arena::CreateMessage<TCreateShardResponse>(req->Arena());
            auto v = result.Extract();
            resp->SetLeaderHost(std::move(v.LeaderHost));
            resp->SetAssignedToHost(std::move(v.AssignedToHost));
            resp->SetShardId(std::move(v.ShardId));
            resp->SetNumId(v.NumId);
            req->SendReply(resp);
        } catch (...) {
            req->SendError(grpc::INTERNAL, "internal error: " + CurrentExceptionMessage());
        }
    }

private:
    TActorId ShardManager_;
};

} // namespace

IActor* CreateCreateShardHandler(TActorId shardManager) {
    return new TCreateShardHandler{shardManager};
}

} // namespace NSolomon::NIngestor::NApi
