#include <solomon/services/ingestor/api/ingestor_grpc_service.grpc.pb.h>
#include <solomon/services/ingestor/lib/shard_actor/shard_actor.h>
#include <solomon/services/ingestor/lib/shard_manager/shard_manager.h>

#include <solomon/libs/cpp/cache/ttl.h>
#include <solomon/libs/cpp/grpc/server/handler.h>
#include <solomon/libs/cpp/proto_convert/format.h>
#include <solomon/libs/cpp/proto_convert/labels.h>

using namespace NActors;

using yandex::monitoring::ingestor::TPulledDataRequest;
using yandex::monitoring::ingestor::TDataProcessResponse;

namespace NSolomon::NIngestor::NApi {
namespace {

class TProcessPulledDataHandler: public NGrpc::TBaseHandler<TProcessPulledDataHandler> {
public:
    TProcessPulledDataHandler(TActorId shardManager) noexcept
        : ShardManager_(shardManager)
        , ShardActorsCache_{TDuration::Seconds(10), 1000u}
    {
    }

public:
    STATEFN(HandleEvent) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TShardManagerEvents::TFindShardResult, OnFindShardResult);
            hFunc(TShardActorEvents::TProcessingResult, OnProcessingResult);
        }
    }

    EMode HandleRequest(NGrpc::TRequestState* req) {
        auto* msg = req->GetMessage<TPulledDataRequest>();

        if (auto shardActor = ShardActorsCache_.Find(msg->GetNumId(), TActivationContext::Now())) {
            SendRequestToShard(*shardActor, req);
            return EMode::Async;
        }

        Send(ShardManager_, new TShardManagerEvents::TFindShard{msg->GetNumId()}, 0, req->ToCookie());
        return EMode::Async;
    }

    void OnFindShardResult(const TShardManagerEvents::TFindShardResult::TPtr& ev) {
        auto req = NGrpc::TRequestState::FromCookie(ev->Cookie);
        if (req->IsReplied()) {
            // request was already expired
            return;
        }

        auto* msg = req->GetMessage<TPulledDataRequest>();
        auto shardActor = ev->Get()->ShardActor;
        if (!shardActor) {
            req->SendError(grpc::NOT_FOUND, TStringBuilder{} << "unknown shard, id=" << msg->GetNumId());
            return;
        }

        ShardActorsCache_.Insert(msg->GetNumId(), shardActor, TActivationContext::Now());
        SendRequestToShard(shardActor, req.release());
    }

    void SendRequestToShard(TActorId shardActor, NGrpc::TRequestState* req) noexcept {
        auto* msg = req->GetMessage<TPulledDataRequest>();
        try {
            auto event = std::make_unique<TShardActorEvents::TProcessData>(SelfId(), msg->GetNumId());
            for (int i = 0; i < msg->host_size(); i++) {
                auto data = std::make_unique<TShardData>();
                data->NumId = msg->GetNumId();
                data->Host = msg->GetHost(i);
                data->HostOptLabels = ReadLabelsFromProto(msg->GetOptionalLabels(i));
                data->MetricsFormat = ConvertProtoFormatToFormat(msg->GetFormat(i));
                data->TimesMillis = TInstant::MilliSeconds(msg->GetResponseTimeMillis(i));

                // TODO: make request pointer non constant
                if (auto* response = const_cast<TPulledDataRequest*>(msg)->MutableResponse(i)) {
                    data->Data = std::move(*response);
                }

                data->SourceId = msg->GetSourceId(i);
                event->Data.push_back(std::move(data));
            }

            // cleanup request's data from the arena ASAP
            req->Arena()->Reset();

            Send(shardActor, event.release(), 0, req->ToCookie());
        } catch (...) {
            req->SendError(grpc::INTERNAL, "internal error: " + CurrentExceptionMessage());
        }
    }

    void OnProcessingResult(const TShardActorEvents::TProcessingResult::TPtr& ev) {
        if (ev->Cookie == 0) {
            // there is no request to reply to
            return;
        }

        auto req = NGrpc::TRequestState::FromCookie(ev->Cookie);
        if (req->IsReplied()) {
            // request was already expired
            return;
        }

        auto* resp = google::protobuf::Arena::CreateMessage<TDataProcessResponse>(req->Arena());
        for (auto&& status: ev->Get()->Statuses) {
            resp->AddStatus(status.StatusType);
            resp->AddErrorMessage(std::move(status.Message));
            resp->AddSuccessMetricCount(status.MetricsWritten);
        }
        req->SendReply(resp);
    }

private:
    TActorId ShardManager_;
    TTtlCache<TNumId, TActorId> ShardActorsCache_;
};

} // namespace

IActor* CreateProcessPulledDataHandler(TActorId shardManager) {
    return new TProcessPulledDataHandler{shardManager};
}

} // namespace NSolomon::NIngestor::NApi
