#include <solomon/services/ingestor/lib/api/service.h>
#include <solomon/services/ingestor/lib/shard_manager/shard_manager.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>
#include <solomon/libs/cpp/clients/ingestor/ingestor_client.h>

#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/gtest/gtest.h>

using namespace NActors;
using namespace NSolomon;
using namespace NIngestor;

using yandex::solomon::config::rpc::TGrpcClientConfig;
using yandex::solomon::config::rpc::TGrpcServerConfig;

THolder<NGrpc::TGRpcServer> CreateGrpcServer(const TGrpcServerConfig& config) {
    NGrpc::TServerOptions opts;

    opts.SetHost("localhost")
        .SetPort(config.GetPort(0))
        .SetWorkerThreads(1);

    return MakeHolder<NGrpc::TGRpcServer>(opts);
}

struct TApiTest: public ::testing::Test {
protected:
    void SetUp() override {
        Runtime_ = TTestActorRuntime::CreateInited();
        Runtime_->UpdateCurrentTime(TInstant::Now());
        Edge_ = Runtime_->AllocateEdgeActor();
        Port_ = NTesting::GetFreePort();
        Service_ = NApi::CreateIngestorService(*Runtime_->SingleSys(), Metrics_, 0, Edge_);

        TGrpcServerConfig serverConf;
        serverConf.AddPort(Port_);
        Server_ = CreateGrpcServer(serverConf);
        Server_->AddService(Service_);
        Server_->Start();

        TGrpcClientConfig clientConf;
        clientConf.AddAddresses("localhost:" + ToString<ui16>(Port_));
        Client_ = CreateIngestorGrpcClient(clientConf, Metrics_);
    }

    void TearDown() override {
        Server_->Stop();
    }

protected:
    NMonitoring::TMetricRegistry Metrics_;
    THolder<TTestActorRuntime> Runtime_;
    TActorId Edge_;
    NTesting::TPortHolder Port_;
    TIntrusivePtr<NGrpc::IGRpcService> Service_;
    THolder<NGrpc::TGRpcServer> Server_;
    IIngestorClientPtr Client_;
};

TEST_F(TApiTest, CreateShard) {
    yandex::monitoring::ingestor::TCreateShardRequest req;
    req.SetProjectId("foo");
    req.SetServiceName("bar");
    req.SetClusterName("baz");

    auto f = Client_->CreateShard(req);
    auto ev = Runtime_->GrabEdgeEvent<TShardManagerEvents::TEvCreateShard>(Edge_);

    TCreateShardResult expected{
        .LeaderHost = "host",
        .AssignedToHost = "another.host",
        .ShardId = "foo_bar_baz",
        .NumId = 23,
    };
    Runtime_->Send(
            ev->Get()->HandlerId,
            MakeHolder<TShardManagerEvents::TEvCreateShardResult>(expected),
            0,
            ev->Cookie);

    auto result = f.ExtractValueSync();
    ASSERT_TRUE(result.Success());

    auto v = result.Extract();
    ASSERT_EQ(v.NumId, expected.NumId);
    ASSERT_EQ(v.Leader, expected.LeaderHost);
    ASSERT_EQ(v.AssignedToHost, expected.AssignedToHost);
    ASSERT_EQ(v.ShardId, expected.ShardId);
}

TEST_F(TApiTest, ErrorCreatingShard) {
    yandex::monitoring::ingestor::TCreateShardRequest req;
    req.SetProjectId("foo");
    req.SetServiceName("bar");
    req.SetClusterName("baz");

    auto f = Client_->CreateShard(req);
    auto ev = Runtime_->GrabEdgeEvent<TShardManagerEvents::TEvCreateShard>(Edge_);

    Runtime_->Send(
            ev->Get()->HandlerId,
            MakeHolder<TShardManagerEvents::TEvCreateShardResult>(TApiCallError{"something went wrong"}),
            0,
            ev->Cookie);

    auto result = f.ExtractValueSync();
    ASSERT_TRUE(result.Fail());
}
