#include "coremon_sync.h"

#include <solomon/services/ingestor/lib/holders/local_shards_holder.h>
#include <solomon/services/ingestor/lib/shard_manager/shard_manager.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/clients/coremon/coremon_client.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/actorid.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/log.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/generic/ptr.h>
#include <util/generic/yexception.h>
#include <util/string/builder.h>
#include <util/string/split.h>
#include <util/system/compiler.h>
#include <util/system/env.h>
#include <util/system/hostname.h>

#include <utility>

namespace NSolomon::NIngestor {
namespace {

using namespace NActors;
using namespace NMonitoring;
using namespace NSolomon::NCoremon;
using yandex::solomon::config::rpc::TGrpcClientConfig;

struct TCoremonSyncPrivateEvents: private TPrivateEvents {
    enum {
        EvCallback = SpaceBegin,
        EvOnUpdateRequest,
        End,
    };

    static_assert(End < SpaceEnd, "too many event types");

    struct TEvCallback: NActors::TEventLocal<TEvCallback, EvCallback> {
        TEvCallback(TErrorOr<NCoremon::TShardAssignments, TApiCallError> shardAssignments)
            : ShardAssignments(std::move(shardAssignments))
        {
        }

        TErrorOr<NCoremon::TShardAssignments, TApiCallError> ShardAssignments;
    };

    struct TEvOnUpdateRequest: NActors::TEventLocal<TEvOnUpdateRequest, EvOnUpdateRequest> {
    };

};

class TCoremonSync: public TActorBootstrapped<TCoremonSync> {
public:
    TCoremonSync(
            ICoremonClusterClientPtr coremonClient,
            TActorId shardManagerId)
        : Coremon_(std::move(coremonClient))
        , ShardManagerId_(shardManagerId)
        , IsCoremonClientRunning_(false)
    {
        TString rawIngestorHost = GetEnv("INGESTOR_HOST");
        if (rawIngestorHost.empty()) {
            Host_ = GetFQDNHostName();
        } else {
            Host_ = rawIngestorHost;
        }

        auto&& rawIngestorShards = GetEnv("INGESTOR_SHARDS");
        for (auto&& shard: StringSplitter(rawIngestorShards).Split(',').SkipEmpty()) {
            BasicShardList_.push_back(FromString<int>(shard.Token()));
        }
    }

    void Bootstrap(const TActorContext &ctx) {
        Become(&TThis::StateWork);
        OnUpdateRequest(ctx);
        MON_DEBUG(CoremonSync, "Coremon Sync start for host " << Host_);
    }

    STFUNC(StateWork) {
        switch (ev->GetTypeRewrite()) {
            cFunc(TEvents::TSystem::Poison, OnPoison);
            hFunc(TCoremonSyncPrivateEvents::TEvCallback, OnCallback);
            CFunc(TCoremonSyncPrivateEvents::EvOnUpdateRequest, OnUpdateRequest);
        };
    }

    STFUNC(StateDying) {
        Y_UNUSED(ctx);

        switch (ev->GetTypeRewrite()) {
            cFunc(TCoremonSyncPrivateEvents::EvCallback, OnDying);
            IgnoreFunc(TCoremonSyncPrivateEvents::TEvOnUpdateRequest);
        };
    }

private:
    void RunCoremonClient(const TActorContext &ctx) {
        if (IsCoremonClientRunning_) {
            MON_ERROR(CoremonSync, "Trying to run more than one connection to Coremon");
            return;
        }

        IsCoremonClientRunning_ = true;
        auto currentCoremonClient = Coremon_->Get(CurrentLeader_);
        if (currentCoremonClient == nullptr) {
            currentCoremonClient = Coremon_->GetAny();
        }

        currentCoremonClient->GetShardAssignments().Subscribe([
            actorSystemPtr = ctx.ExecutorThread.ActorSystem,
            self = ctx.SelfID]
            (auto f) {
            try {
                auto v = f.ExtractValue();
                actorSystemPtr->Send(self, new TCoremonSyncPrivateEvents::TEvCallback(std::move(v)));
            } catch (...) {
                MON_DEBUG_C(*actorSystemPtr, CoremonSync, "Fetch Shard Data error: " << CurrentExceptionMessage());
            }
        });
    }

    void OnPoison() {
        if (!IsCoremonClientRunning_) {
            PassAway();
            return;
        }
        Become(&TThis::StateDying);
    }

    void OnDying() {
        IsCoremonClientRunning_ = false;
        PassAway();
    }

    void OnUpdateRequest(const TActorContext &ctx) {
        if (!IsCoremonClientRunning_) {
            RunCoremonClient(ctx);
        }

        ctx.Schedule(
            GetRandomTime(RELOAD_SHARDS_BASE_DURATION, RELOAD_SHARDS_VARIATION),
            new TCoremonSyncPrivateEvents::TEvOnUpdateRequest);
    }

    void OnCallback(const TCoremonSyncPrivateEvents::TEvCallback::TPtr& request) {
        IsCoremonClientRunning_ = false;
        if (request->Get()->ShardAssignments.Fail()) {
            MON_ERROR(CoremonSync, "Fetch Shard Data error: " << CurrentExceptionMessage());
            return;
        }

        const auto& shardAssignments = request->Get()->ShardAssignments.Extract();
        CurrentLeader_ = shardAssignments.Leader;
        TIntrusivePtr<TLocalShardsHolder> localShards = new TLocalShardsHolder();
        for (auto const& [hostInfo, shardList]: shardAssignments.Assignments) {
            if (hostInfo.first == Host_) {
                localShards->LocalShards.insert(shardList.begin(), shardList.end());
            }
        }

        for (const auto& shardId: BasicShardList_) {
            localShards->LocalShards.insert(shardId);
        }

        Send(ShardManagerId_, new TShardManagerEvents::TEvUpdateShardAssignments(localShards));
    }

    // TODO(semenovma): Extract this function to ingestor/lib module
    TDuration GetRandomTime(const TDuration& base, const TDuration& variation) {
        return TDuration::Seconds(RandomNumber(variation.Seconds())) + base;
    }

    ICoremonClusterClientPtr Coremon_;
    TList<TShardId> BasicShardList_;
    TActorId ShardManagerId_;
    TMetricRegistry Registry_;
    TString Host_;
    TString CurrentLeader_;
    bool IsCoremonClientRunning_;

    constexpr static const TDuration RELOAD_SHARDS_BASE_DURATION = TDuration::Seconds(3);
    constexpr static const TDuration RELOAD_SHARDS_VARIATION = TDuration::Seconds(6);
};

} // namespace

NActors::IActor* CreateCoremonSync(
        NCoremon::ICoremonClusterClientPtr coremonClient,
        TActorId shardManagerId) {
    return new TCoremonSync(std::move(coremonClient), shardManagerId);
}

} // namespace NSolomon::NIngestor
