#include "request_tracer.h"
#include "request_tracer_actor.h"

#include <solomon/libs/cpp/config/units.h>

#include <library/cpp/monlib/counters/meter.h>

#include <util/generic/hash_set.h>
#include <util/random/random.h>

using namespace NSolomon;
using namespace NMonitoring;
using namespace NActors;
using namespace yandex::monitoring::ingestor;

namespace NSolomon::NIngestor {

struct TRequestTracerMetrics {
    explicit TRequestTracerMetrics(TMetricRegistry& registry) {
        WriteBytes = registry.Rate({{"sensor", "ingestor.tracing.write.bytes"}});
        WriteRequests = registry.Rate({{"sensor", "ingestor.tracing.write.requests"}});
        DropBytes = registry.Rate({{"sensor", "ingestor.tracing.drop.bytes"}});
        DropRequests = registry.Rate({{"sensor", "ingestor.tracing.drop.requests"}});
    }

    IRate* WriteBytes;
    IRate* WriteRequests;
    IRate* DropBytes;
    IRate* DropRequests;
};

class TRequestTracer: public IRequestTracer {
public:
    TRequestTracer(TActorSystem& actorSystem, TActorId actorId, TRequestTracerMetrics metrics, double quantile, double bytesRateLimit, THashSet<ui32> numIds)
        : ActorSystem_(actorSystem)
        , ActorID_(actorId)
        , Metrics_(std::move(metrics))
        , NumIds_(std::move(numIds))
        , Quantile_(quantile)
        , BytesRateLimit_(bytesRateLimit)
        , BytesRate_(NMonitoring::TMovingAverage::OneMinute())
    {
    }

    void Trace(const TPulledDataRequest& req) override {
        if (NeedTrace(req)) {
            auto event = std::make_unique<TRequestTracerEvent::TTracePull>(req);
            ActorSystem_.Send(ActorID_, event.release());
        }
    }

    void Trace(const TPushedDataRequest& req) override {
        if (NeedTrace(req)) {
            auto event = std::make_unique<TRequestTracerEvent::TTracePush>(req);
            ActorSystem_.Send(ActorID_, event.release());
        }
    }

private:
    TActorSystem& ActorSystem_;
    TActorId ActorID_;
    TRequestTracerMetrics Metrics_;
    const THashSet<ui32> NumIds_;
    const double Quantile_;
    const double BytesRateLimit_;
    TMovingAverage BytesRate_;

    void Dropped(ui64 bytes) {
        Metrics_.DropRequests->Inc();
        Metrics_.DropBytes->Add(bytes);
    }

    void Written(ui64 bytes) {
        Metrics_.WriteRequests->Inc();
        Metrics_.WriteBytes->Add(bytes);
        BytesRate_.Update(bytes);
    }

    template <typename T>
    bool NeedTrace(const T req) {
        if (!NumIds_.empty() && !NumIds_.contains(req.numid())) {
            return false;
        }

        if (Quantile_ < RandomNumber<double>()) {
            return false;
        }

        if (BytesRateLimit_ != 0.0 && BytesRateLimit_ <= BytesRate_.GetRate()) {
            Dropped(req.ByteSizeLong());
            return false;
        }

        Written(req.ByteSizeLong());
        return true;
    }
};

IRequestTracerPtr CreateRequestTracer(
    const RequestTracerConfig& conf,
    TActorRuntime& actorRuntime,
    TMetricRegistry& registry)
{
    auto actor = CreateRequestTracerActor(conf.file());
    auto actorId = actorRuntime.Register(actor.release());
    return CreateRequestTracer(conf, actorRuntime.ActorSystem(), actorId, registry);
}

IRequestTracerPtr CreateRequestTracer(
    const RequestTracerConfig& conf,
    TActorSystem& actorSystem,
    TActorId actorId,
    TMetricRegistry& registry)
{
    THashSet<ui32> numIds(conf.num_id().begin(), conf.num_id().end());
    double bytesRateLimit = static_cast<double>(FromProtoDataSize(conf.bytes_rate_limit()));
    TRequestTracerMetrics metrics{registry};

    return new TRequestTracer(
        actorSystem,
        actorId,
        std::move(metrics),
        conf.trace_quantile(),
        bytesRateLimit,
        std::move(numIds));
}
} // namespace NSolomon::NIngestor
