#include "request_tracer_actor.h"

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

#include <library/cpp/logger/file.h>
#include <library/cpp/logger/record.h>

#include <util/string/builder.h>

#include <google/protobuf/messagext.h>

using namespace NActors;
using namespace NSolomon::NIngestor;

namespace NSolomon::NIngestor {

enum ERequestType {
    PULL = 0,
    PUSH = 1,
};

class TRequestTracerActor: public TActor<TRequestTracerActor> {
public:
    explicit TRequestTracerActor(const TString& path)
        : TActor(&TRequestTracerActor::StateNormal)
        , File_(path)
    {
    }

    STATEFN(StateNormal) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TRequestTracerEvent::TTracePush, OnTracePush);
            hFunc(TRequestTracerEvent::TTracePull, OnTracePull);
            hFunc(TEvents::TEvPoison, OnPoison);
        }
    }

    void OnTracePush(TRequestTracerEvent::TTracePush::TPtr& ev) {
        auto request = std::move(ev->Get()->Request);
        Write(ERequestType::PUSH, request);
        if (ev->Get()->SendReply) {
            Send(ev->Sender, new TRequestTracerEvent::TTraceResp);
        }
    }

    void OnTracePull(TRequestTracerEvent::TTracePull::TPtr& ev) {
        auto request = std::move(ev->Get()->Request);
        Write(ERequestType::PULL, request);
        if (ev->Get()->SendReply) {
            Send(ev->Sender, new TRequestTracerEvent::TTraceResp);
        }
    }

    void OnPoison(TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new TEvents::TEvPoisonTaken{});
        PassAway();
    }

    void Write(ERequestType type, const ::google::protobuf::Message& message) {
        TStringBuilder builder;
        builder.reserve(message.ByteSizeLong() + 4u);
        {
            google::protobuf::io::StringOutputStream so(&builder);
            google::protobuf::io::CodedOutputStream co(&so);
            co.WriteVarint32(type);
            co.WriteVarint32(message.ByteSize());
            Y_ENSURE(message.SerializeToCodedStream(&co), "cannot serialize message");
        }
        File_.WriteData(TLogRecord{ELogPriority::TLOG_INFO, builder.Data(), builder.size()});
    }

private:
    TFileLogBackend File_;
};

std::unique_ptr<IActor> CreateRequestTracerActor(const TString& path) {
    return std::make_unique<TRequestTracerActor>(path);
}
} // namespace NSolomon::NIngestor
