#include <solomon/services/ingestor/lib/requests_tracer/request_tracer_actor.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/actors/core/event.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/system/fs.h>
#include <util/system/tempfile.h>

#include <memory>
#include <fstream>
#include <google/protobuf/io/zero_copy_stream_impl.h>

using namespace NSolomon;
using namespace NMonitoring;
using namespace NIngestor;
using namespace yandex::monitoring::ingestor;

class TRequestTracerActorTest: public ::testing::Test {
protected:
    void SetUp() override {
        ActorRuntime_ = TTestActorRuntime::CreateInited(1, false, false);

        auto tracer = CreateRequestTracerActor(TempFile_.Name());
        TracerId_ = ActorRuntime_->Register(tracer.release());
        EdgeId_ = ActorRuntime_->AllocateEdgeActor();
    }

    void TearDown() override {
        ActorRuntime_.Reset();
    }

    void SendRequest(TPulledDataRequest request) const {
        auto ev = std::make_unique<TRequestTracerEvent::TTracePull>(std::move(request));
        ev->SendReply = true;
        ActorRuntime_->Send(TracerId_, EdgeId_, THolder(ev.release()));
    }

    void SendRequest(TPushedDataRequest request) const {
        auto ev = std::make_unique<TRequestTracerEvent::TTracePush>(std::move(request));
        ev->SendReply = true;
        ActorRuntime_->Send(TracerId_, EdgeId_, THolder(ev.release()));
    }

    template <typename TEvent>
    typename TEvent::TPtr ReceiveResponse() {
        return ActorRuntime_->GrabEdgeEvent<TEvent>(EdgeId_, TDuration::Seconds(10));
    }

protected:
    THolder<TTestActorRuntime> ActorRuntime_;
    NActors::TActorId EdgeId_;
    NActors::TActorId TracerId_;
    TTempFile TempFile_{"tmplogfile"};
};

TEST_F(TRequestTracerActorTest, TracePull) {
    auto request = TPulledDataRequest();
    request.set_numid(42);
    request.add_response("one two tree");
    SendRequest(request);

    ReceiveResponse<TRequestTracerEvent::TTraceResp>();
    TLogBackend::ReopenAllBackends(true);

    std::ifstream in(TempFile_.Name());
    google::protobuf::io::IstreamInputStream ii(&in);
    google::protobuf::io::CodedInputStream ci(&ii);

    {
        ui32 type;
        ui32 size;
        TPulledDataRequest result;
        ASSERT_TRUE(ci.ReadVarint32(&type));
        ASSERT_EQ(0u, type);
        ASSERT_TRUE(ci.ReadVarint32(&size));
        ASSERT_TRUE(result.ParseFromCodedStream(&ci));

        ASSERT_EQ(result.numid(), request.numid());

        ASSERT_EQ(result.response_size(), request.response_size());
        for (int i = 0; i < result.response_size(); i++) {
            ASSERT_EQ(result.response(i), request.response(i)) << "i=" << i;
        }
    }
}

TEST_F(TRequestTracerActorTest, TracePush) {
    auto request = TPushedDataRequest();
    request.set_numid(42);
    request.set_content("one two tree");
    SendRequest(request);

    ReceiveResponse<TRequestTracerEvent::TTraceResp>();
    TLogBackend::ReopenAllBackends(true);

    std::ifstream in(TempFile_.Name());
    google::protobuf::io::IstreamInputStream ii(&in);
    google::protobuf::io::CodedInputStream ci(&ii);

    {
        ui32 type;
        ui32 size;
        TPushedDataRequest result;
        ASSERT_TRUE(ci.ReadVarint32(&type));
        ASSERT_EQ(1u, type);
        ASSERT_TRUE(ci.ReadVarint32(&size));
        ASSERT_TRUE(result.ParseFromCodedStream(&ci));

        ASSERT_EQ(result.numid(), request.numid());
        ASSERT_EQ(result.content(), request.content());
    }
}
