#include <solomon/services/ingestor/lib/requests_tracer/request_tracer.h>
#include <solomon/services/ingestor/lib/requests_tracer/request_tracer_actor.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/actors/core/event.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/system/fs.h>

#include <memory>

using namespace NSolomon;
using namespace NMonitoring;
using namespace NIngestor;
using namespace yandex::monitoring::ingestor;

class TRequesterTracerTest: public ::testing::Test {
protected:
    void SetUp() override {
        ActorRuntime_ = TTestActorRuntime::CreateInited(1, false, false);
        EdgeId_ = ActorRuntime_->AllocateEdgeActor();
    }

    void TearDown() override {
        ActorRuntime_.Reset();
    }

    IRequestTracerPtr Tracer(RequestTracerConfig& config) {
        auto *actorSystem = ActorRuntime_->GetAnyNodeActorSystem();
        return CreateRequestTracer(config, *actorSystem, EdgeId_, Registry_);
    }

    template <typename TEvent>
    typename TEvent::TPtr ReceiveEvent() {
        return ActorRuntime_->GrabEdgeEvent<TEvent>(EdgeId_);
    }

protected:
    THolder<TTestActorRuntime> ActorRuntime_;
    NActors::TActorId EdgeId_;
    NMonitoring::TMetricRegistry Registry_;
};

TEST_F(TRequesterTracerTest, SkipAllEvents) {
    RequestTracerConfig config;
    config.set_trace_quantile(0);

    auto tracer = Tracer(config);
    for (int index = 0; index < 10'000; index++) {
        TPulledDataRequest req;
        req.set_numid(42);
        req.add_response("should be ignored");
        tracer->Trace(req);
    }

    auto* writeRequests = Registry_.Rate({{"sensor", "ingestor.tracing.write.requests"}});
    ASSERT_EQ(0u, writeRequests->Get());
}

TEST_F(TRequesterTracerTest, FilterByNumId) {
    RequestTracerConfig config;
    config.add_num_id(42);
    config.add_num_id(24);
    config.set_trace_quantile(1);

    auto tracer = Tracer(config);
    auto* writeRequests = Registry_.Rate({{"sensor", "ingestor.tracing.write.requests"}});
    ASSERT_EQ(0u, writeRequests->Get());

    {
        TPulledDataRequest req;
        req.set_numid(42);
        req.add_response("should be traced");
        tracer->Trace(req);

        auto ev = ReceiveEvent<TRequestTracerEvent::TTracePull>();
        ASSERT_EQ(req.numid(), ev->Get()->Request.numid());
        ASSERT_EQ(req.response_size(), ev->Get()->Request.response_size());
        for (int i = 0; i < req.response_size(); i++) {
            ASSERT_EQ(req.response(i), ev->Get()->Request.response(i)) << "i=" << i;
        }
        ASSERT_EQ(1u, writeRequests->Get());
    }
    {
        TPulledDataRequest ignored;
        ignored.set_numid(43);
        ignored.add_response("should be ignored");
        tracer->Trace(ignored);
        ASSERT_EQ(1u, writeRequests->Get());
    }
    {
        TPulledDataRequest req;
        req.set_numid(24);
        req.add_response("should be also traced");
        tracer->Trace(req);

        auto ev = ReceiveEvent<TRequestTracerEvent::TTracePull>();
        ASSERT_EQ(req.numid(), ev->Get()->Request.numid());
        ASSERT_EQ(req.response_size(), ev->Get()->Request.response_size());
        for (int i = 0; i < req.response_size(); i++) {
            ASSERT_EQ(req.response(i), ev->Get()->Request.response(i)) << "i=" << i;
        }
        ASSERT_EQ(2u, writeRequests->Get());
    }
}

TEST_F(TRequesterTracerTest, TraceAll) {
    RequestTracerConfig config;
    config.set_trace_quantile(1);

    auto tracer = Tracer(config);
    auto* writeRequests = Registry_.Rate({{"sensor", "ingestor.tracing.write.requests"}});
    ASSERT_EQ(0u, writeRequests->Get());

    for (ui64 index = 1; index < 10; index++) {
        TPulledDataRequest req;
        req.set_numid(index);
        req.add_response("should be traced");
        tracer->Trace(req);

        auto ev = ReceiveEvent<TRequestTracerEvent::TTracePull>();
        ASSERT_EQ(req.numid(), ev->Get()->Request.numid());
        ASSERT_EQ(req.response_size(), ev->Get()->Request.response_size());
        for (int i = 0; i < req.response_size(); i++) {
            ASSERT_EQ(req.response(i), ev->Get()->Request.response(i)) << "i=" << i;
        }
        ASSERT_EQ(index, writeRequests->Get());
    }
}

TEST_F(TRequesterTracerTest, TracePercent) {
    RequestTracerConfig config;
    config.set_trace_quantile(0.5);

    auto tracer = Tracer(config);
    for (int index = 0; index < 10'000; index++) {
        TPulledDataRequest req;
        req.set_numid(42);
        req.add_response("download content");
        tracer->Trace(req);
    }

    auto* writeRequests = Registry_.Rate({{"sensor", "ingestor.tracing.write.requests"}});
    ASSERT_NEAR(5'000.0, static_cast<double>(writeRequests->Get()), 1'000.0);
}
