#pragma once

#include "metric_transformer.h"

#include <solomon/libs/cpp/sync/rw_lock.h>

#include <library/cpp/monlib/metrics/summary_collector.h>
#include <library/cpp/monlib/metrics/labels.h>
#include <library/cpp/monlib/metrics/metric_type.h>

#include <util/generic/vector.h>

#include <unordered_map>

namespace NSolomon::NIngestor {

class TAggrState {
public:
    using TLabels = NMonitoring::TLabels;

    struct TAggregate {
        TLabels Labels;
        NMonitoring::ISummaryDoubleSnapshotPtr Aggregate;
        NMonitoring::EMetricType MetricType;

        TAggregate(TLabels labels, NMonitoring::ISummaryDoubleSnapshotPtr snapshot, NMonitoring::EMetricType type)
            : Labels(std::move(labels))
            , Aggregate(std::move(snapshot))
            , MetricType(type)
        {
        }
    };

    using EMetricType = NMonitoring::EMetricType;

    struct TLabelsHasher {
        ui64 operator()(const TLabels& labels) const {
            return labels.Hash();
        }
    };

    struct TAggrsTableValue {
        NMonitoring::ISummaryDoubleCollectorPtr Collector;
        NMonitoring::EMetricType MetricType;

        size_t SizeBytes() const {
            return sizeof(*this) + Collector->SizeBytes();
        }
    };

    class TAggrsTable {
    public:
        using TIter = std::unordered_map<TLabels, TAggrsTableValue, TLabelsHasher>::iterator;
        using TConstIter = std::unordered_map<TLabels, TAggrsTableValue, TLabelsHasher>::const_iterator;

        TAggrsTable() = default;

        TAggrsTable(TAggrsTable&& other)
            : Value_(std::move(other.Value_))
            , SizeBytes_(std::move(other.SizeBytes_))
        {
        }

        size_t Size() const {
            return Value_.size();
        }

        TConstIter Cbegin() const {
            return Value_.cbegin();
        }

        TIter begin() {
            return Value_.begin();
        }

        TConstIter Cend() const {
            return Value_.cend();
        }

        TIter end() {
            return Value_.end();
        }

        auto Extract(TConstIter iter) {
            SizeBytes_ -= SizeBytes(iter->first);
            SizeBytes_ -= iter->second.SizeBytes();
            return Value_.extract(iter);
        }

        auto Emplace(TLabels key, TAggrsTableValue val) {
            auto res = Value_.emplace(std::move(key), std::move(val));
            if (res.second) {
                SizeBytes_ += SizeBytes(res.first->first);
                SizeBytes_ += res.first->second.SizeBytes();
            }
            return res;
        }

        TConstIter Find(const TLabels& val) const {
            return Value_.find(val);
        }

        TIter Find(const TLabels& val) {
            return Value_.find(val);
        }

        size_t SizeBytes() const {
            Y_VERIFY(SizeBytes_ < (1l << 34));
            return SizeBytes_;
        }

    private:
        std::unordered_map<TLabels, TAggrsTableValue, TLabelsHasher> Value_;
        size_t SizeBytes_ = sizeof(*this);

        static size_t SizeBytes(const TLabels& labels) {
            size_t res = 0;
            for (const auto& l: labels) {
                auto* label = static_cast<const NMonitoring::TLabel*>(&l);
                res += label->NameStr().capacity();
                res += label->ValueStr().capacity();
                res += sizeof(*label);
            }
            res += (labels.capacity() - labels.size()) * sizeof(NMonitoring::TLabel);
            return res;
        }
    };

public:
    TAggrState(const TVector<TAggrRuleItem>& aggrRules)
        : Transforemer_(std::move(aggrRules))
    {
    }

    void CollectTransforme(const TLabels& labels, NMonitoring::EMetricType metricType, double value) {
        auto transformedLabels = Transforemer_.Transform(labels);
        if (!transformedLabels.Empty()) {
            Collect(transformedLabels, metricType, value);
        }
    }

    void Collect(const TLabels& labels, NMonitoring::EMetricType metricType, double value) {
        {
            auto table = Table_.Read();
            if (auto it = table->Find(labels); it != table->Cend()) {
                it->second.Collector->Collect(value);
                return;
            }
        }

        NMonitoring::ISummaryDoubleCollectorPtr collector = MakeHolder<NMonitoring::TSummaryDoubleCollector>();
        auto table = Table_.Write();
        auto [it, _] = table->Emplace(labels, TAggrsTableValue{std::move(collector), metricType});
        it->second.Collector->Collect(value);
    }

    TVector<TAggregate> ReleaseAggregates() {
        auto table = Table_.Write();
        TVector<TAggregate> aggregates;
        aggregates.reserve(table->Size());

        for (auto it = table->Cbegin(); it != table->Cend();) {
            auto toExtract = it++;
            auto node = table->Extract(toExtract);
            aggregates.emplace_back(
                    std::move(node.key()),
                    node.mapped().Collector->Snapshot(),
                    node.mapped().MetricType);
        }
        return aggregates;
    }

    TAggrsTable ReleaseAggrsTable() {
        auto table = Table_.Write();
        return std::move(*table);
    }

    size_t SizeBytes() const {
        size_t res = sizeof(*this);
        res += Table_.Read()->SizeBytes();
        res += Transforemer_.SizeBytes();
        Y_VERIFY(res < (1l << 34));
        return res;
    }

private:
    NSync::TLightRwLock<TAggrsTable> Table_;
    TMetricTransformer Transforemer_;
};

} // namespace NSolomon::NIngestor
