#include "data_processor.h"
#include "yasm_aggregator.h"

#include <solomon/libs/cpp/slog/slog.h>

#include <library/cpp/threading/future/async.h>
#include <library/cpp/monlib/encode/json/json.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>
#include <library/cpp/monlib/encode/prometheus/prometheus.h>

#include <util/stream/buffer.h>
#include <util/stream/mem.h>

using NMonitoring::ECompression;
using NMonitoring::EFormat;
using NMonitoring::ETimePrecision;
using NMonitoring::EMetricType;
using NSolomon::NSlog::CreateSlogEncoder;
using TProcessingResult = NSolomon::NIngestor::TDataProcessor::TProcessingResult;
using NMonitoring::ISummaryDoubleSnapshotPtr;
using yandex::solomon::common::UrlStatusType;

namespace NSolomon::NIngestor {
namespace {

TProcessingStatus ClassifyError() {
    TProcessingStatus status{};
    status.StatusType = UrlStatusType::UNKNOWN_STATUS_TYPE;
    status.MetricsWritten = 0;

    try {
        // exception dispatcher (https://isocpp.org/wiki/faq/exceptions#throw-without-an-object)
        throw;
    } catch (const NMonitoring::TJsonDecodeError& err) {
        status.StatusType = UrlStatusType::JSON_ERROR;
        status.Message = err.AsStrBuf();
    } catch (const NMonitoring::TSpackDecodeError& err) {
        status.StatusType = UrlStatusType::SPACK_ERROR;
        status.Message = err.AsStrBuf();
    } catch (const TQuotaError& err) {
        status.StatusType = UrlStatusType::QUOTA_ERROR;
        status.Message = err.AsStrBuf();
    } catch (const TParseError& err) {
        status.StatusType = UrlStatusType::PARSE_ERROR;
        status.Message = err.AsStrBuf();
    } catch (...) {
        status.StatusType = UrlStatusType::UNKNOWN_ERROR;

        // TODO(ivanzhukov@): SOLOMON-5558
        Cerr << "couldn't parse and validate data: " << CurrentExceptionMessage();
    }

    return status;
}

} // namespace

class TAggregatesProcessor {
public:
    TAggregatesProcessor(TInstant intervalStart, TDuration intervalLength_, ILogWriter* writer)
        : IntervalStart_(intervalStart)
        , IntervalLength_(intervalLength_)
        , Writer_(writer)
        , MetricsWritten_(0u)
    {
    }

    void Open() {
        Writer_->OnStreamBegin();
        Writer_->OnStep(IntervalLength_);
        Writer_->OnCommonTime(IntervalStart_);
    }

    void Close() {
        Writer_->OnStreamEnd();
        Writer_->Close();
    }

    void OnYasmSummaryAggregate(TAggregate<NMonitoring::ISummaryDoubleSnapshotPtr> aggr) {
        auto point = MakeAggrPoint(std::move(aggr.Aggregate), aggr.MetricType, aggr.Count);
        FlushAggregate(std::move(aggr.Labels), std::move(point), NMonitoring::EMetricType::DSUMMARY);
    }

    void OnYasmHistogramAggregate(TAggregate<NMonitoring::IHistogramSnapshotPtr> aggr) {
        auto point = MakeAggrPoint(std::move(aggr.Aggregate), aggr.MetricType, aggr.Count);
        FlushAggregate(std::move(aggr.Labels), std::move(point), NMonitoring::EMetricType::HIST);
    }

    void OnYasmLogHistogramAggregate(TAggregate<NMonitoring::TLogHistogramSnapshotPtr> aggr) {
        auto point = MakeAggrPoint(std::move(aggr.Aggregate), aggr.MetricType, aggr.Count);
        FlushAggregate(std::move(aggr.Labels), std::move(point), NMonitoring::EMetricType::LOGHIST);
    }

    ui64 MetricsWritten() const {
        return MetricsWritten_;
    }

private:
    void FlushAggregate(TLabels&& labels,
            TAggrPointWithType&& point,
            NMonitoring::EMetricType aggrType)
    {
        Writer_->OnMetricBegin(aggrType);

        Writer_->OnLabelsBegin();
        for (size_t i = 0; i < labels.size(); ++i) {
            Writer_->OnLabel(labels.Name(i), labels.Value(i));
        }

        Writer_->OnLabelsEnd();

        auto flags =  static_cast<ELogFlagsComb>(ELogFlags::Merge) |  static_cast<ELogFlagsComb>(ELogFlags::Count);
        if (point.GetDenom()) {
            flags |= static_cast<ELogFlagsComb>(ELogFlags::Denom);
        }

        Writer_->OnPoint(flags, std::move(point));

        Writer_->OnMetricEnd();

        ++MetricsWritten_;
    }

    template <class TSnapshot>
    TAggrPointWithType MakeAggrPoint(TSnapshot&& snapshot, EMetricType metricType, size_t count) {
        TAggrPointWithType point(IntervalStart_, snapshot.Get());
        point.SetCount(count);
        if (IsRate(metricType)) {
            point.SetDenom(IntervalLength_.MilliSeconds());
        }
        return point;
    }

    bool IsRate(EMetricType type) {
        return type == EMetricType::RATE || type == EMetricType::HIST_RATE;
    }

private:
    TInstant IntervalStart_;
    TDuration IntervalLength_;
    ILogWriter* Writer_;
    ui64 MetricsWritten_;
};

class TYasmAggregatesConsumer: public IYasmAggregatesProcessorConsumer {
public:
    TYasmAggregatesConsumer(TAggregatesProcessor& processor)
        : Processor_(processor)
    {
    }

    void OnSummary(TAggregate<NMonitoring::ISummaryDoubleSnapshotPtr> aggr) override {
        Processor_.OnYasmSummaryAggregate(std::move(aggr));
    }

    void OnHistogram(TAggregate<NMonitoring::IHistogramSnapshotPtr> aggr) override {
        Processor_.OnYasmHistogramAggregate(std::move(aggr));
    }

    void OnLogHistogram(TAggregate<NMonitoring::TLogHistogramSnapshotPtr> aggr) override {
        Processor_.OnYasmLogHistogramAggregate(std::move(aggr));
    }

private:
    TAggregatesProcessor& Processor_;
};

auto TDataProcessor::MakeYasmAggrFunc(TInstant intervalStart, TDuration intervalLength) {
    auto aggregates = YasmAggrState_->ReleaseAggregates();

    auto func = [this,
                 aggregates = std::move(aggregates),
                 intervalStart,
                 intervalLength]() mutable {
        TProcessingResult metaAndData;
        TStringOutput dataOut(metaAndData.Data);
        TStringOutput metaOut(metaAndData.Meta);
        auto slogWriter = CreateSlogEncoder(
                ShardId_,
                NMonitoring::ETimePrecision::SECONDS,
                NMonitoring::ECompression::LZ4,
                &dataOut,
                &metaOut);
        TAggregatesProcessor aggrProcessor(intervalStart, intervalLength, slogWriter.Get());
        TYasmAggregatesConsumer consumer(aggrProcessor);
        try {
            aggrProcessor.Open();
            aggregates->Consume(consumer);
            ui32 metricsWritten = aggrProcessor.MetricsWritten();
            metaAndData.Statuses.push_back({UrlStatusType::OK, TString{}, metricsWritten});
            aggrProcessor.Close();
        } catch (...) {
            // TODO: Classify error
            metaAndData.Statuses.push_back({UrlStatusType::UNKNOWN_ERROR, TString{}, 0});
            Cerr << "exception while processing yasm aggregates: " << CurrentExceptionMessage();
        }
        return metaAndData;
    };

    return func;
}

NThreading::TFuture<TProcessingResult> TDataProcessor::ProcessAggregates(
        TInstant intervalStart,
        TDuration intervalLength)
{
    if (IsYasmShard_) {
        return NThreading::Async(MakeYasmAggrFunc(intervalStart, intervalLength), *Executor_);
    } else {
        Cerr << "unsupported request (solomon mode not supported) " << ShardId_ << "\n";
        auto promise = NThreading::NewPromise<TProcessingResult>();
        promise.SetValue(TProcessingResult{});
        return promise.GetFuture();
    }
}

NThreading::TFuture<TProcessingResult> TDataProcessor::GetMetaAndData(
        TString data,
        TMetricProcessorOptions opts,
        NMonitoring::EFormat metricsFormat) const
{
    auto func = [this, data = std::move(data), opts = std::move(opts), metricsFormat]() mutable {
        TErrorListenerCounter errors;

        TProcessingResult metaAndData;
        TStringOutput dataOut(metaAndData.Data);
        TStringOutput metaOut(metaAndData.Meta);
        auto slogWriter = CreateSlogEncoder(
                ShardId_,
                NMonitoring::ETimePrecision::SECONDS,
                NMonitoring::ECompression::LZ4,
                &dataOut,
                &metaOut);

        opts.YasmAggrState = YasmAggrState_.Get();
        opts.Errors = &errors;

        slogWriter->OnStreamBegin();
        metaAndData.Statuses.push_back(DecodeData(data, opts, metricsFormat, slogWriter.Get()));
        slogWriter->OnStreamEnd();
        slogWriter.Reset(); // will flush all buffered data

        OnDataProcessed_();
        return metaAndData;
    };

    return NThreading::Async(std::move(func), *Executor_);
}

NThreading::TFuture<TDataProcessor::TProcessingResult> TDataProcessor::ProcessBatch(
        std::vector<std::unique_ptr<TShardData>> data,
        TInstant intervalStart,
        TDuration intervalLength)
{
    auto func = [this, data = std::move(data), intervalStart, intervalLength]() mutable {
        TProcessingResult result;
        result.Statuses.resize(data.size());

        TStringOutput dataOut(result.Data);
        TStringOutput metaOut(result.Meta);

        auto slogWriter = CreateSlogEncoder(
            ShardId_,
            NMonitoring::ETimePrecision::SECONDS,
            NMonitoring::ECompression::LZ4,
            &dataOut,
            &metaOut);

        slogWriter->OnStreamBegin();

        for (size_t i = 0; i < data.size(); i++) {
            auto shardData = std::move(data[i]);
            if (!shardData) {
                continue;
            }

            NMonitoring::TLabels hostLabels;
            if (!shardData->IsPush && !shardData->Host.empty()) {
                hostLabels.Add(TStringBuf{"host"}, shardData->Host);
            }

            TErrorListenerCounter errors;

            TMetricProcessorOptions opts{
                .CommonLabels = std::move(hostLabels),
                .CommonOptLabels = std::move(shardData->HostOptLabels),
                .ResponseTs = shardData->TimesMillis,
                .IsShardMemOnly = IsMemonlyShard_,
                .IntervalStart = intervalStart,
                .IntervalLength = intervalLength,
                .LabelPool = &LabelPool_
            };


            opts.YasmAggrState = YasmAggrState_.Get();
            opts.Errors = &errors;

            result.Statuses[i] = DecodeData(shardData->Data, opts, shardData->MetricsFormat, slogWriter.Get());
        }

        slogWriter->OnStreamEnd();
        slogWriter.Reset(); // will flush all buffered data

        OnDataProcessed_();
        return result;
    };

    return NThreading::Async(std::move(func), *Executor_);
}

TProcessingStatus TDataProcessor::DecodeData(
        const TString& data,
        TMetricProcessorOptions metricProcessorOptions,
        NMonitoring::EFormat metricsFormat,
        ILogWriter* writer)
{
    auto metricProcessor = CreateBatchMetricProcessor(std::move(metricProcessorOptions), writer);
    TProcessingStatus status;

    try {
        switch (metricsFormat) {
            case EFormat::SPACK: {
                TMemoryInput memIn{data.c_str(), data.size()};
                DecodeSpackV1(&memIn, metricProcessor.Get());
                break;
            }
            case EFormat::JSON:
                DecodeJson({data.c_str(), data.size()}, metricProcessor.Get());
                break;
            case EFormat::PROMETHEUS:
                NMonitoring::DecodePrometheus({data.c_str(), data.size()}, metricProcessor.Get());
                break;
    //        case EFormat::MONITORING:
    //            DecodeJsonMonitoring({data.c_str(), data.size()}, metricProcessor.Get());
            default:
                throw TParseError() << "unsupported format: " << metricsFormat;
        }

        status = metricProcessor->Status();
    } catch (...) {
        status = ClassifyError();
    }

    return status;
}

} // namespace NSolomon::NIngestor
