#pragma once

#include "metric_processor.h"
#include "prev_values_supplier.h"
#include "aggregation_state.h"
#include "yasm_aggregator.h"
#include "shard_data.h"

#include <solomon/services/ingestor/lib/shard_config/shard_config.h>
#include <solomon/services/ingestor/config/ingestor_config.pb.h>

#include <library/cpp/monlib/encode/format.h>
#include <library/cpp/threading/future/future.h>

#include <util/generic/buffer.h>
#include <util/thread/pool.h>
#include <util/generic/vector.h>

#include <functional>

namespace NSolomon::NIngestor {

class TErrorListenerCounter: public IProcessingErrorListener {
public:
    TErrorListenerCounter()
        : Counter_((ui32)EInvalidMetricReason::ERRORS_NUMBER, 0u)
        , ErrorsNumber_(0u)
    {
    }

    void InvalidMetric(EInvalidMetricReason invalidMetricReason) override {
        ++ErrorsNumber_;
        ++Counter_[(ui32)invalidMetricReason];
    }

    bool AnyErrors() const noexcept {
        return ErrorsNumber_ != 0;
    }

    ui32 ErrorsCount(EInvalidMetricReason invalidMetricReason) const noexcept {
        return Counter_[(ui32)invalidMetricReason];
    }

    TString GetMessage() const override {
        if (ErrorsNumber_ == 0) {
            return {};
        }
        TStringBuilder builder;
        builder << "Total:" << ErrorsNumber_ << " errors. {\n";
        for (size_t i = 0; i < static_cast<size_t>(EInvalidMetricReason::ERRORS_NUMBER); ++i) {
            auto count = Counter_[i];
            if (count) {
                builder << "   " << InvalidMetricReasonToStr(static_cast<EInvalidMetricReason>(i))
                        << ":" << count << ",\n";
            }
        }
        builder << "}";
        return builder;
    }

private:
    TVector<ui32> Counter_;
    ui32 ErrorsNumber_;
};

class TDataProcessor {
public:
    struct TProcessingResult {
        TString Data;
        TString Meta;
        std::vector<TProcessingStatus> Statuses;
    };

    TDataProcessor(
            std::shared_ptr<IThreadPool> executor,
            TShardId id,
            bool isYasmShard,
            bool isMemonlyShard,
            TVector<TAggrRuleItem> aggrRules,  // NOLINT(performance-unnecessary-value-param): false positive
            TVector<TYasmAggrRule> yasmAggrRules,  // NOLINT(performance-unnecessary-value-param): false positive
            TLabelPool& labelPool,
            std::function<void()> onDataProcessed)
        : Executor_(std::move(executor))
        , ShardId_(id)
        , IsYasmShard_(isYasmShard)
        , IsMemonlyShard_{isMemonlyShard}
        , LabelPool_(labelPool)
        , OnDataProcessed_(std::move(onDataProcessed))
        , PrevValues_(MakeHolder<TPrevValuesSupplier>(DerivMaxAllowedSkips_, DerivDeleteMetricsPeriod_))
        , AggrState_(MakeHolder<TAggrState>(std::move(aggrRules)))
        , YasmAggrState_(IsYasmShard_ ? MakeHolder<TYasmAggrState>(std::move(yasmAggrRules)) : nullptr)
    {
    }

    void CloseInterval() {
        PrevValues_->CloseInterval();
    }

    static TProcessingStatus DecodeData(
            const TString& data,
            TMetricProcessorOptions metricProcessorOptions,
            NMonitoring::EFormat metricsFormat,
            ILogWriter* writer);

    NThreading::TFuture<TProcessingResult> GetMetaAndData(
            TString data,
            TMetricProcessorOptions metricProcessorOptions,
            NMonitoring::EFormat metricsFormat) const;

    NThreading::TFuture<TProcessingResult> ProcessAggregates(TInstant intervalStart, TDuration IntervalLength);

    NThreading::TFuture<TProcessingResult> ProcessBatch(
        std::vector<std::unique_ptr<TShardData>> data,
        TInstant intervalStart,
        TDuration intervalLength);

public:
    static constexpr size_t DerivMaxAllowedSkips_ = 2;
    static constexpr size_t DerivDeleteMetricsPeriod_ = 10;

    size_t SizeBytes() const {
        size_t res = sizeof(*this);
        if (PrevValues_) {
            res += PrevValues_->SizeBytes();
        }
        if (AggrState_) {
            res += AggrState_->SizeBytes();
        }
        if (YasmAggrState_) {
            res += YasmAggrState_->SizeBytes();
        }
        Y_VERIFY(res < (1l << 34));
        return res;
    }
private:
    std::shared_ptr<IThreadPool> Executor_;
    TShardId ShardId_;
    bool IsYasmShard_;
    bool IsMemonlyShard_;

    TLabelPool& LabelPool_;

    std::function<void()> OnDataProcessed_;

    THolder<TPrevValuesSupplier> PrevValues_;
    THolder<TAggrState> AggrState_;
    THolder<TYasmAggrState> YasmAggrState_;

private:
    auto MakeSolomonAggrFunc(TInstant intervalStart, TDuration intervalLength);

    auto MakeYasmAggrFunc(TInstant intervalStart, TDuration intervalLength);
};

} // namespace NSolomon::NIngestor
