#pragma once

#include <solomon/libs/cpp/intern/str_pool.h>
#include <util/generic/string.h>
#include <util/generic/hash.h>
#include <util/generic/vector.h>

#include <util/generic/vector.h>

#include <utility>
#include <algorithm>

#include <type_traits>
#include <functional>

namespace NSolomon::NIngestor {

constexpr TStringBuf CONTAINER_TAG = "yasm_container";

using TIntLabel = std::pair<NIntern::TStringId , NIntern::TStringId>;

inline TIntLabel MakeIntLabel(NIntern::TStringId name, NIntern::TStringId value) {
    return std::make_pair(name, value);
}

class TLabelPool: public NIntern::TRCUStringPool {
public:
    TLabelPool();

    const NIntern::TStringId HostIntTag;
    const NIntern::TStringId GroupIntTag;
    const NIntern::TStringId AggrIntTag;
    const NIntern::TStringId ContainerIntTag;
    const NIntern::TStringId SignalIntTag;
};

class TLabels: public TVector<TIntLabel> {
public:
    using TBase = TVector<TIntLabel>;

    explicit TLabels(TLabelPool* labelPool);
    TLabelPool* LabelPool() const;

    template <class TName, class TValue>
    void Add(TName, TValue) {
        static_assert(
                std::is_same_v<TName, TStringBuf> || std::is_same_v<TName, NIntern::TStringId>,
                "label name/value must be either TStringId or TStringBuf"
            );
        static_assert(
                std::is_same_v<TValue, TStringBuf> || std::is_same_v<TValue, NIntern::TStringId>,
                "label name/value must be either TStringId or TStringBuf"
            );
    }

    template <>
    void Add<NIntern::TStringId, NIntern::TStringId>(NIntern::TStringId name, NIntern::TStringId value) {
        emplace_back(MakeIntLabel(name, value));
    }

    template <>
    void Add<TStringBuf, NIntern::TStringId>(TStringBuf name, NIntern::TStringId value) {
        emplace_back(MakeIntLabel(LabelPool_->Intern(name), value));
    }

    template <>
    void Add<NIntern::TStringId, TStringBuf>(NIntern::TStringId name, TStringBuf value) {
        emplace_back(MakeIntLabel(name, LabelPool_->Intern(value)));
    }

    template <>
    void Add<TStringBuf, TStringBuf>(TStringBuf name, TStringBuf value) {
        emplace_back(MakeIntLabel(LabelPool_->Intern(name), LabelPool_->Intern(value)));
    }


    TStringBuf Name(size_t index) const;
    TStringBuf Value(size_t index) const;

    TString AsString() const;

    bool operator==(const TLabels& other) const;

    size_t SizeBytes() const {
        return sizeof(*this) + sizeof(*LabelPool_);
    }

private:
    TLabelPool* LabelPool_;
};

} // namespace NSolomon::NIngestor

template <>
struct THash<NSolomon::NIngestor::TLabels> {
    ui64 operator()(const NSolomon::NIngestor::TLabels& la) const {
        ui64 hash = 0;
        for (auto [name, value]: la) {
            hash = CombineHashes(THash<ui64>()(((ui64) name << 32u) | value), hash);
        }

        return hash;
    }
};

template <>
struct std::hash<NSolomon::NIngestor::TLabels> {
    ui64 operator()(const NSolomon::NIngestor::TLabels& la) const {
        return THash<NSolomon::NIngestor::TLabels>()(la);
    }
};
