#include "metric_processor.h"
#include "labels.h"

#include <solomon/libs/cpp/exception/exception.h>
#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/labels/validate.h>
#include <solomon/libs/cpp/timeseries/timeseries.h>
#include <solomon/services/ingestor/lib/type_utils/histogram_utils.h>

#include <library/cpp/monlib/metrics/labels.h>
#include <library/cpp/monlib/metrics/metric_value.h>

#include <util/datetime/base.h>

namespace NSolomon::NIngestor {
namespace {

#define PARSE_ENSURE(COND, ...) MON_ENSURE_EX(COND, TParseError() << __VA_ARGS__)

using NMonitoring::EMetricType;

constexpr ui64 MAX_LABELS_COUNT = 16;
constexpr ui64 ADDITIONAL_FUTURE_LABELS = 3; // "project", "cluster", "service"

// TODO: update selfmon error metrics
bool IsLabelValid(TValidationMode mode, TStringBuf name, TStringBuf value/*, ErrorListener errors*/) {
    if (TValidationMode::LegacySkip == mode) {
        return true;
    } else if (TValidationMode::StrictSkip == mode || TValidationMode::StrictFail == mode) {
        // use strict validation rules for new metrics
        bool isNameValid = TLabelsValidatorStrict::ValidateLabelNameSafe(name);
        bool isValueValid = TLabelsValidatorStrict::ValidateLabelValueSafe(value);

        if (!(isNameValid && isValueValid)) {
            // TODO: support errors metrics
            // errors.invalidMetric(InvalidMetricReason.INVALID_LABEL);

            if (TValidationMode::StrictFail == mode) {
                // TODO:
                // throw new UrlStatusTypeException(UrlStatusType.PARSE_ERROR);
            }

            return false;
        }
    } else {
        ythrow yexception() << "unknown validation mode: " << mode;
    }

    return true;
}

const ui64 NOT_BEFORE = TInstant::ParseIso8601("2000-01-01T00:00:00Z").MilliSeconds();
const ui64 NOT_AFTER = TInstant::ParseIso8601("2038-01-19T03:14:07Z").MilliSeconds();

bool IsGoodMillis(TInstant ts) {
    const auto& millis = ts.MilliSeconds();
    return millis >= NOT_BEFORE && millis <= NOT_AFTER;
}

void CheckTs(TInstant ts) {
    if (!IsGoodMillis(ts)) {
        throw TParseError() << "invalid timestamp " << ts.Seconds();
    }
}


//bool IsDeriv(EMetricType kind) {
//    return EMetricType::RATE == kind || EMetricType::HIST_RATE == kind;
//}

// constexpr ui64 DEFAULT_DENOM = 0;

enum class EStreamMode {
    ONE_SHOT,
    BATCH,
};

// TODO: how to test every state change? (multiple OnMetricBegin() are not allowed, etc.)
// TODO: can this consumer instance be used multiple times in a row?
template <EStreamMode StreamMode>
class TMetricProcessor: public IMetricProcessor {
public:
    explicit TMetricProcessor(TMetricProcessorOptions opts, ILogWriter* writer)
        : Options_(std::move(opts))
        , Writer_(writer)
        , State_(Options_.LabelPool)
    {
    }

private:
    void OnStreamBegin() override {
        if constexpr (StreamMode == EStreamMode::ONE_SHOT) {
            Writer_->OnStreamBegin();
        }

        Writer_->OnStep(Options_.IntervalLength);
    }

    void OnStreamEnd() override {
        if (StreamMode == EStreamMode::ONE_SHOT) {
            Writer_->OnStreamEnd();
            Reset();
        }
    }

    void OnCommonTime(TInstant time) override {
        State_.CommonTs = time;
    }

    void OnMetricBegin(EMetricType kind) override {
        if (Options_.Quota.MaxMetricsPerUrl > 0 && (MetricCount_ + 1) > Options_.Quota.MaxMetricsPerUrl) {
            throw TQuotaError() << "more than " << Options_.Quota.MaxMetricsPerUrl << " metrics from one URL";
        }

        State_.Type = kind;

        if (kind != EMetricType::UNKNOWN) {
            State_.Initialized = true;
        }
    }

    void OnMetricEnd() override {
        if (State_.Series.Empty() || !State_.Initialized) {
            State_.Reset();
            return;
        }

        Flush();

        ++MetricCount_;
        State_.Reset();
    }

    void OnLabelsBegin() override {
    }

    void OnLabel(TStringBuf name, TStringBuf value) override {
        if (!State_.Initialized) {
            return;
        }

        // check that lately we can add 3 more labels for project, cluster, service
        if (State_.Labels.size() + 1 + ADDITIONAL_FUTURE_LABELS > MAX_LABELS_COUNT) {
            throw TParseError() << "too many labels";
        }

        if (!IsLabelValid(Options_.ValidationMode, name, value)) {
            Options_.Errors->InvalidMetric(EInvalidMetricReason::INVALID_LABEL);
            State_.Reset();
            return;
        }

        State_.Labels.emplace_back(
            MakeIntLabel(Options_.LabelPool->Intern(name), Options_.LabelPool->Intern(value))
        );
    }

    void OnLabelsEnd() override {
        if (State_.Labels.empty() || !State_.Initialized) {
            State_.Initialized = false;
            return;
        }

        RemoveYasmContainer(State_.Labels);
        SetTransformed();
        SetGroups();
        SetHost();

        std::sort(State_.Labels.begin(), State_.Labels.end());
    }

    void OnDouble(TInstant time, double value) override {
        OnValue(time, value);
    }

    void OnInt64(TInstant time, i64 value) override {
        OnValue(time, value);
    }

    void OnUint64(TInstant time, ui64 value) override {
        OnValue(time, value);
    }

    void OnHistogram(TInstant time, NMonitoring::IHistogramSnapshotPtr snapshot) override {
        OnValue(time, snapshot.Get());
    }

    void OnSummaryDouble(TInstant time, NMonitoring::ISummaryDoubleSnapshotPtr snapshot) override {
        OnValue(time, snapshot.Get());
    }

    void OnLogHistogram(TInstant time, NMonitoring::TLogHistogramSnapshotPtr snapshot) override {
        OnValue(time, snapshot.Get());
    }

    TProcessingStatus Status() override {
        if (Options_.Errors) {
            Status_.Message = Options_.Errors->GetMessage();
        }

        Status_.StatusType = yandex::solomon::common::UrlStatusType::OK;
        Status_.MetricsWritten = MetricCount_;

        return Status_;
    }

private:
    void RemoveYasmContainer(TLabels& labels) {
        auto container = std::find_if(labels.begin(), labels.end(), [this](TIntLabel label) {
            return label.first == Options_.LabelPool->ContainerIntTag;
        });

        if (container != labels.end()) {
            State_.Host = container->second;
            std::iter_swap(container, labels.rbegin());
            labels.pop_back();
        }
    }

    void SetTransformed() {
        auto transformer = Options_.YasmAggrState->GetTransformer();

        State_.Transformed = transformer->Transform(State_.Labels);
    }

    void SetGroups() {
        for (const auto& label: Options_.CommonOptLabels) {
            if (label.Name().StartsWith(GROUP_TAG_PREFIX)) {
                State_.Groups.emplace_back(Options_.LabelPool->Intern(label.Value()));
            }
        }
    }

    void SetHost() {
        if (State_.Host != NIntern::InvalidStringId) {
            return;
        }

        auto mb = Options_.CommonLabels.Find(NLabels::LABEL_HOST);
        if (mb) {
            State_.Host = Options_.LabelPool->Intern(mb->Value());
        }
    }

    void SetAndValidateTime(TInstant& time) const {
        if (!time) {
            time = State_.CommonTs ? State_.CommonTs : Options_.ResponseTs;
        }

        CheckTs(time);
    }

    bool MatchInterval(TInstant time) const {
        return time >= Options_.IntervalStart && time < Options_.IntervalStart + Options_.IntervalLength;
    }

    bool IsGroupShard() const {
        return !State_.Groups.empty();
    }

    EMetricType YasmMetricType(EMetricType type) {
        switch (type) {
            case NMonitoring::EMetricType::HIST:
            case NMonitoring::EMetricType::LOGHIST:
            case NMonitoring::EMetricType::DSUMMARY:
                return type;

            case NMonitoring::EMetricType::COUNTER:
            case NMonitoring::EMetricType::GAUGE:
            case NMonitoring::EMetricType::IGAUGE:
                return NMonitoring::EMetricType::DSUMMARY;

            case NMonitoring::EMetricType::UNKNOWN:
            case NMonitoring::EMetricType::RATE:
            case NMonitoring::EMetricType::HIST_RATE:
                ythrow yexception() << "unexpected type: " << NMonitoring::MetricTypeToStr(type);
        }
    }

    template <class TFunc>
    void ApplyOnce(TFunc func, TLabels& labels, NIntern::TStringId group = NIntern::InvalidStringId) {
        if (IsGroupShard()) {
            labels.emplace_back(
                    MakeIntLabel(Options_.LabelPool->HostIntTag, Options_.LabelPool->AggrIntTag)
                );
        } else {
            labels.emplace_back(
                    MakeIntLabel(Options_.LabelPool->HostIntTag, State_.Host)
                );
        }

        if (group != NIntern::InvalidStringId) {
            labels.emplace_back(
                    MakeIntLabel(Options_.LabelPool->GroupIntTag, group)
                );
        }

        func(labels);

        labels.pop_back();
        if (group != NIntern::InvalidStringId) {
            labels.pop_back();
        }
    }

    template <class TFunc>
    void Apply(TFunc func) {
        auto it = State_.Transformed.begin();
        while (it != State_.Transformed.end()) {
            auto node = State_.Transformed.extract(it++);

            if (IsGroupShard()) {
                for (auto group: State_.Groups) {
                    ApplyOnce(func, node.value(), group);
                }
            } else {
                ApplyOnce(func, node.value());
            }
        }

        if (IsGroupShard()) {
            for (auto group: State_.Groups) {
                ApplyOnce(func, State_.Labels, group);
            }
        }
    }

    template <class TValue>
    void OnValue(TInstant time, TValue value) {
        if (!State_.Initialized) {
            return;
        }

        SetAndValidateTime(time);

        if (MatchInterval(time)) {
            auto collect = [this, value](const TLabels& labels) {
                Options_.YasmAggrState->Collect(value, labels, State_.Type);
            };

            if (Options_.YasmAggrState) {
                Apply(collect);
                State_.Transformed.clear();
            }

            if (IsGroupShard()) {
                return;
            }
        }

        if constexpr (std::is_arithmetic_v<TValue>) {
            State_.Series.Add(time, GaugeToSummary(State_.Labels, value).Get(), 0, IsGroupShard());
        } else {
            State_.Series.Add(time, value, 0, IsGroupShard());
        }
    }

    void WriteMetric(const TLabels& labels, ELogFlagsComb flags, const TAggrTimeSeries& series) {
        if (series.Empty()) {
            return;
        }

        Writer_->OnMetricBegin(YasmMetricType(State_.Type));

        Writer_->OnLabelsBegin();
        for (const auto& [name, value]: labels) {
            Writer_->OnLabel(
                    Options_.LabelPool->Find(name),
                    Options_.LabelPool->Find(value));
        }

        Writer_->OnLabelsEnd();

        Writer_->OnTimeSeries(flags, series);

        Writer_->OnMetricEnd();
    }

    void Flush() {
        State_.Series.SortByTs();

        auto flags = static_cast<ELogFlagsComb>(ELogFlags::None);

        auto write = [this, &flags](const TLabels& labels) {
            WriteMetric(labels, flags, State_.Series);
        };

        if (!IsGroupShard()) {
            ApplyOnce(write, State_.Labels);
        }

        flags |= static_cast<ELogFlagsComb>(ELogFlags::Merge) |  static_cast<ELogFlagsComb>(ELogFlags::Count);

        Apply(write);
    }

    void Reset() {
        State_.Reset();
        MetricCount_ = 0;
    }

private:
    static constexpr TStringBuf GROUP_TAG_PREFIX = "group_";

    struct TState {
        TState(TLabelPool* labelPool)
            : Labels(labelPool)
        {
        }

        TLabels Labels;
        TLabelsSet Transformed;

        TAggrTimeSeries Series;
        EMetricType Type{EMetricType::UNKNOWN};
        NIntern::TStringId Host{NIntern::InvalidStringId};
        std::vector<NIntern::TStringId> Groups;
        TInstant CommonTs;

        bool Initialized{false};

        void Reset() {
            Labels.clear();
            Series.Clear();
            Transformed.clear();

            Type = EMetricType::UNKNOWN;

            CommonTs = TInstant::Zero();

            Host = NIntern::InvalidStringId;
            Groups.clear();

            Initialized = false;
        }
    };

private:
    TMetricProcessorOptions Options_;
    ILogWriter* Writer_;

    TState State_;
    TProcessingStatus Status_;
    ui32 MetricCount_{0};
};

#undef PARSE_ENSURE

} // namespace

IMetricProcessorPtr CreateMetricProcessor(
        NSolomon::NIngestor::TMetricProcessorOptions opts,
        ILogWriter* writer)
{
    return MakeHolder<TMetricProcessor<EStreamMode::ONE_SHOT>>(std::move(opts), writer);
}

IMetricProcessorPtr CreateBatchMetricProcessor(
    NSolomon::NIngestor::TMetricProcessorOptions opts,
    ILogWriter* writer)
{
    return MakeHolder<TMetricProcessor<EStreamMode::BATCH>>(std::move(opts), writer);
}

} // namespace NSolomon::NIngestor
