#include "metric_transformer.h"

#include <solomon/libs/cpp/glob/glob.h>
#include <solomon/libs/cpp/labels/validate.h>

#include <util/string/split.h>
#include <util/string/strip.h>
#include <util/string/escape.h>
#include <util/string/split.h>

#include <util/stream/format.h>

#include <util/generic/hash_set.h>

using NMonitoring::TLabels;
using NMonitoring::TLabel;

namespace NSolomon::NIngestor {

using TLabelBackend = TLabel::TStringType;

template <typename TStringType>
class TLabelNameHash {
public:
    size_t operator()(TStringBuf name) const {
        return Hasher_(name);
    }

    size_t operator()(const NMonitoring::TLabelImpl<TStringType>& label) const {
        return Hasher_(TStringBuf{label.Name()});
    }

private:
    THash<TStringBuf> Hasher_;
};

template <typename TStringType>
class TLabelNameEqual {
public:
    bool operator()(TStringBuf name, const NMonitoring::TLabelImpl<TStringType>& label) const {
        return name == TStringBuf{label.Name()};
    }

    bool operator()(const NMonitoring::TLabelImpl<TStringType>& label, TStringBuf name) const {
        return this->operator()(name, label);
    }

    bool operator()(TStringBuf l, TStringBuf r) const {
        return l == r;
    }

    bool operator()(const NMonitoring::TLabelImpl<TStringType>& l, const NMonitoring::TLabelImpl<TStringType>& r) const {
        return l.Name() == r.Name();
    }
};

using TLabelsHashMap = THashMap<TLabelBackend, TLabelBackend, TLabelNameHash<TLabelBackend>, TLabelNameEqual<TLabelBackend>>;

static std::optional<TStringBuf> GetLabelValueSafe(TStringBuf) {
    return std::nullopt;
}

template <class... TLabelsContainer>
static std::optional<TStringBuf> GetLabelValueSafe(TStringBuf name, const TLabelsHashMap& first, const TLabelsContainer&... rest) {
    if (auto it = first.find(name); it != first.end()) {
        return it->second;
    }
    return GetLabelValueSafe(name, rest...);
}

template <class... TLabelsContainer>
static TStringBuf GetLabelValue(TStringBuf name, const TLabelsHashMap& first, const TLabelsContainer&... rest) {
    auto res = GetLabelValueSafe(name, first, rest...);
    if (!res.has_value()) {
        ythrow yexception() << "haven't found a label with the name \"" << name << "\""
                            << " inside metric labels";
    }
    return res.value();
}

template <class TFunc>
static TLabel ParseLabel(TStringBuf str, TFunc nameValidator, TFunc valueValidator) {
    TLabelBackend name;
    TLabelBackend value;
    TLabelBackend* token = &name;
    str = StripString(str);
    for (size_t i = 0; i < str.size(); ++i) {
        char ch = str[i];
        switch (ch) {
            case '\\': {
                Y_ENSURE(i != str.size() - 1, "unclosed escaping");
                auto nextCh = str[i + 1];
                if (nextCh == '{' || nextCh == '}') {
                    // Propagate braces to later LabelValueExpr validation and parsing
                    *token += '\\';
                    *token += nextCh;
                } else {
                    UnescapeC(TStringBuf{str.data() + i, 2}, *token);
                }
                ++i;
                break;
            }
            case ' ':
                ythrow yexception() << "unescaped space characters";
            case '=':
                Y_ENSURE(token != &value,
                         "multiple '=' inside a label value. Either remove it or escape it with '\\'");
                Y_ENSURE(!name.empty(), "empty label name");
                token = &value;
                break;
            default:
                *token += ch;
                break;
        }
    }

    nameValidator(name);
    valueValidator(value);

    return {name, value};
}

static bool IsPrintableAndNotEmpty(TStringBuf str) {
    if (str.empty()) {
        return false;
    }
    for (auto& ch: str) {
        if (!(ch >= 32 && ch <= 126)) {
            return false;
        }
    }
    return true;
}

static void ValidateMatchLabelName(TStringBuf name) {
    Y_ENSURE(IsPrintableAndNotEmpty(name), "Label name(=" << HexText(name) << ") in a Match is empty or has non-printable characters");
}

static void ValidateMatchLabelValue(TStringBuf value) {
    // Labels gathered from different data sources: Prometheus Exporter, etc.
    // A value can hold almost any character, so there's no real pattern to test
    Y_ENSURE(IsPrintableAndNotEmpty(value), "Label value(=" << HexText(value) << ") in a Match is empty or has non-printable characters");
}

static void ValidateExternalLabelName(TStringBuf name) {
    Y_ENSURE(IsPrintableAndNotEmpty(name), "Label name(=" << HexText(name) << ") in a substitution is empty or has non-printable characters");
}

static void ValidateReplaceMetaLabelValue(TStringBuf value) {
    Y_ENSURE(IsPrintableAndNotEmpty(value), "Label value(=" << HexText(value) << ") in a ReplaceMeta is empty or has non-printable characters");
}

static TLabel ParseTarget(TStringBuf str) {
    return ParseLabel(str, TLabelsValidator::ValidateLabelName, ValidateReplaceMetaLabelValue);
}

static TLabel ParseCondition(TStringBuf str) {
    return ParseLabel(str, ValidateMatchLabelName,  ValidateMatchLabelValue);
}

class TValueSelector {
public:
    static TValueSelector ParseFromStr(TStringBuf value) {
        if (value == TStringBuf("*")) {
            return TValueSelector({TMatchCondition(EMatchType::PRESENT, {})}, false);
        }
        TVector<TMatchCondition> conds;
        bool negative = value.StartsWith('!');
        for (auto it: StringSplitter(value.begin() + negative, value.end()).Split('|')) {
                    conds.emplace_back(TMatchCondition(it.Token(), negative));
        }

        return TValueSelector(std::move(conds), negative);
    }

    bool IsMatch(TStringBuf value) const {
        return Negative_ ? AllMatch(value) : AnyMatch(value);
    }

    bool IsAbsent() const {
        bool anyMatch = false;
        for (const auto& matchCond: Conditions_) {
            anyMatch |= matchCond.IsAbsent();
        }
        return anyMatch;
    }

    void CalcSizeBytes() {
        SizeBytes_ = sizeof(*this);
        for (const auto& condition: Conditions_) {
            SizeBytes_ += condition.SizeBytes();
        }
        SizeBytes_ += (Conditions_.capacity() - Conditions_.size()) * sizeof(TMatchCondition);
    }

    size_t SizeBytes() const {
        return SizeBytes_;
    }

private:
    enum class EMatchType {
        GLOB_POSITIVE,
        GLOB_NEGATIVE,
        PRESENT,
        ABSENT
    };

    class TMatchCondition {
    public:
        TMatchCondition(EMatchType matchType, TLabelBackend cond)
            : MatchType_(matchType)
            , Cond_(std::move(cond)) {}

        TMatchCondition(TStringBuf cond, bool negative) {
            if (cond.empty()) {
                ythrow yexception() << "empty condition";
            }
            if (!negative && cond == "*") {
                MatchType_ = EMatchType::PRESENT;
            } else if (!negative && cond == "-") {
                MatchType_ = EMatchType::ABSENT;
            } else if (cond != "-") {
                MatchType_ = negative ? EMatchType::GLOB_NEGATIVE : EMatchType::GLOB_POSITIVE;
                Cond_ = cond;
            } else {
                ythrow yexception() << "invalid condition";
            }
        }

        bool IsMatch(TStringBuf value) const {
            if (MatchType_ == EMatchType::ABSENT) {
                return false;
            }
            if (MatchType_ == EMatchType::PRESENT) {
                return true;
            }
            bool globMatch = IsGlobMatch(Cond_, value);
            return globMatch == (MatchType_ == EMatchType::GLOB_POSITIVE);
        }

        bool IsAbsent() const {
            return MatchType_ == EMatchType::ABSENT;
        }

        size_t SizeBytes() const {
            return sizeof(*this) + Cond_.capacity();
        }

    private:
        EMatchType MatchType_;
        TLabelBackend Cond_;
    };

private:
    size_t SizeBytes_ = sizeof(*this);
    TVector<TMatchCondition> Conditions_;
    bool Negative_;

    explicit TValueSelector(TVector<TMatchCondition> conditions, bool negative)
        : Conditions_(std::move(conditions))
        , Negative_(negative)
    {
        CalcSizeBytes();
    }

    bool AnyMatch(TStringBuf value) const {
        bool anyMatch = false;
        for (const auto& matchCond: Conditions_) {
            anyMatch |= matchCond.IsMatch(value);
        }
        return anyMatch;
    }

    bool AllMatch(TStringBuf value) const {
        for (const auto& matchCond: Conditions_) {
            if (!matchCond.IsMatch(value)) {
                return false;
            }
        }
        return true;
    }
};

class TLabelValueExpr {
public:
    TLabelValueExpr(TStringBuf expr) {
        BuildFrom(expr);
    }

    TVector<TStringBuf> GetNames() const {
        TVector<TStringBuf> result;
        if (ValueNamePairs_.size() < 2) {
            return result;
        }
        for (size_t i = 1; i < ValueNamePairs_.size(); i += 2) {
            if (ValueNamePairs_[i].has_value()) {
                result.emplace_back(ValueNamePairs_[i].value());
            }
        }
        return result;
    }

    template <class... TLabelsContainer>
    // Labels by priority, from high to low
    TLabelBackend Eval(const TLabelsContainer&... labels) const {
        if (ValueNamePairs_.size() == 1) {
            // shortcut
            return ValueNamePairs_[0].value();
        }
        if (ValueNamePairs_.size() == 2 && !ValueNamePairs_[0].has_value()) {
            // shortcut
            auto value = GetLabelValue(ValueNamePairs_[1].value(), labels...);
            return TLabelBackend{value.data(), value.size()};
        }
        TStringBuf value;
        TLabelBackend result;
        for (size_t i = 0; i < ValueNamePairs_.size(); ++i) {
            if (!ValueNamePairs_[i].has_value()) {
                continue;
            }
            if (i % 2 == 0) {
                value = ValueNamePairs_[i].value();
            } else {
                value = GetLabelValue(ValueNamePairs_[i].value(), labels...);
            }
            result.append(value.data(), value.size());
        }
        return result;
    }

    TString ToString() const {
        TString r;
        for (size_t i = 0; i < ValueNamePairs_.size(); ++i) {
            if (i % 2 != 0) {
                r.append("{{");
            }
            if (ValueNamePairs_[i].has_value()) {
                r.append(ValueNamePairs_[i].value());
            }
            if (i % 2 != 0) {
                r.append("}}");
            }
        }
        return r;
    }

    size_t SizeBytes() const {
        return SizeBytes_;
    }

private:
#define EXPR_ERROR(msg, expr, pos) msg << ", at pos " << (pos) << ": \"" << (expr) << "\""

    void BuildFrom(TStringBuf expr) {
        expr = StripString(expr);
        enum {
            NAME,
            VALUE,
        } tokenType = VALUE;
        TLabelBackend token;
        // only braces are left unescaped
        for (size_t i = 0; i != expr.size(); ++i) {
            char ch = expr[i];
            if (ch == '\\') {
                Y_ENSURE(i != expr.size() - 1, "unclosed escaping");
                UnescapeC(TStringBuf{expr.data() + i, 2}, token);
                ++i;
            } else if (ch == ' ' || (ch == '{' && tokenType == NAME) || (ch == '}' && tokenType == VALUE)) {
                ythrow yexception() << EXPR_ERROR("ill-formed substitution. unescaped '" << ch << "' inside a label "
                                                                                         << (tokenType == NAME ? "name": "value"), expr, i);
            } else if ((ch == '{' && tokenType == VALUE) || (ch == '}' && tokenType == NAME)) {
                Y_ENSURE((i != expr.size() - 1) && (expr[i + 1] == ch),
                         EXPR_ERROR("ill-formed substitution. unescaped or unmatched braces '" << ch << "'", expr, i));
                ++i;
                switch (tokenType) {
                    case NAME: AppendName(token); break;
                    case VALUE: AppendValue(token); break;
                }
                token = {};
                tokenType = tokenType == NAME ? VALUE : NAME;
            } else {
                token += ch;
            }
        }
        if (!token.empty()) {
            switch (tokenType) {
                case NAME: ythrow yexception() << EXPR_ERROR("unclosed braces inside a substitution", expr, expr.size() - 1);
                case VALUE: AppendValue(token); break;
            }
        }
    }

#undef EXPR_ERROR

    bool IsLastElementAValue() const {
        return ValueNamePairs_.size() % 2 != 0;
    }

    void AppendValue(TStringBuf value) {
        if (!value.empty()) {
            TLabelsValidatorStrict::ValidateLabelValueSafe(value);
        }
        Y_ENSURE(!IsLastElementAValue(), "two label values cannot be inserted one after another");
        ui64 oldCapacity = ValueNamePairs_.capacity();
        if (value.empty()) {
            ValueNamePairs_.emplace_back(std::nullopt);
        } else {
            ValueNamePairs_.emplace_back(value);
            SizeBytes_ += ValueNamePairs_.back()->capacity();
        }
        ui64 newCapacity = ValueNamePairs_.capacity();
        ui64 diff = newCapacity - oldCapacity;
        if (diff > 0) {
            SizeBytes_ += diff * sizeof(TLabelBackend);
        }
    }

    void AppendName(TStringBuf name) {
        ValidateExternalLabelName(name);
        Y_ENSURE(IsLastElementAValue(), "two label names cannot be inserted one after another");
        ui64 oldCapacity = ValueNamePairs_.capacity();
        ValueNamePairs_.emplace_back(name);
        ui64 newCapacity = ValueNamePairs_.capacity();
        ui64 diff = newCapacity - oldCapacity;
        if (diff > 0) {
            SizeBytes_ += diff * sizeof(TLabelBackend);
        }
        SizeBytes_ += ValueNamePairs_.back()->capacity();
    }

private:
    size_t SizeBytes_ = sizeof(*this);
    TVector<std::optional<TLabelBackend>> ValueNamePairs_;
};

struct TLabelNameWithExpr {
    TLabelBackend Name;
    TLabelValueExpr Expr;
};

class TLabelsTransformer {
public:
    TLabelsTransformer(TLabels&& labels) {
        decltype(LabelsToDrop_) labelsToDrop;
        decltype(LabelsToAdd_) labelsToAdd;
        decltype(LabelsWithSubs_) labelsWithSubs;

        for (auto&& label: labels) {
            if (label.Value() == "-") {
                labelsToDrop.emplace(label.Name());
            } else if (!label.Value().Contains("{{")) {
                labelsToAdd.emplace(label.Name(), label.Value());
            } else {
                labelsWithSubs.emplace_back(TLabelNameWithExpr{
                        TLabelBackend{label.Name().data(), label.Name().size()},
                        label.Value()
                });
            }
        }

        LabelsToDrop_ = std::move(labelsToDrop);
        LabelsToAdd_ = std::move(labelsToAdd);
        LabelsWithSubs_ = std::move(labelsWithSubs);
        CalcSizeBytes();
    }

    void Transform(TLabelsHashMap& labels) {
        // For rules that both change the value and rely on the old one.
        // For instance: "cluster=sdev -> project=newFixedValue, client={{project}}"
        TLabelsHashMap labelNameToOriginalValue;
        for (auto& label: LabelsWithSubs_) {
            if (!StoreLabelsOriginalValues(label.Expr.GetNames(), labels, labelNameToOriginalValue)) {
                // TODO: write to log
                continue;
            }
            TLabelBackend newValue = label.Expr.Eval(labelNameToOriginalValue, labels);
            labels[label.Name] = newValue;
        }

        for (auto& labelName: LabelsToDrop_) {
            labels.erase(labelName);
        }

        for (auto&& it: LabelsToAdd_) {
            labels[it.first] = it.second;
        }
    }

    void CalcSizeBytes() {
        SizeBytes_ = sizeof(*this);
        for (const auto& labelToDrop: LabelsToDrop_) {
            SizeBytes_ += labelToDrop.capacity();
        }
        for (const auto& labelToAdd: LabelsToAdd_) {
            SizeBytes_ += labelToAdd.first.capacity() + labelToAdd.second.capacity();
        }
        for (auto& labelWithSubs: LabelsWithSubs_) {
            SizeBytes_ += labelWithSubs.Expr.SizeBytes() + labelWithSubs.Name.capacity();
        }
        SizeBytes_ += LabelsToDrop_.size() * sizeof(TLabelBackend);
        SizeBytes_ += LabelsToAdd_.size() * (sizeof(TLabelBackend) + sizeof(TLabelBackend));
        SizeBytes_ += (LabelsWithSubs_.capacity() - LabelsWithSubs_.size()) * sizeof(TLabelNameWithExpr);
        SizeBytes_ += LabelsWithSubs_.capacity() * sizeof(TLabelBackend);
    }

    size_t SizeBytes() const {
        return SizeBytes_;
    }

private:
    bool StoreLabelsOriginalValues(
            const TVector<TStringBuf>& names,
            const TLabelsHashMap& labels,
            TLabelsHashMap& labelNameToValue) const
    {
        for (auto& name: names) {
            if (!labels.contains(name)) {
                // TODO: write to log
                return false;
            }
            labelNameToValue[name] = labels.at(name);
        }
        return true;
    }

    TLabelsTransformer(THashSet<TLabelBackend>&& labelsToDrop, TLabelsHashMap&& labelsToAdd, TVector<TLabelNameWithExpr>&& labelsWithSubs)
            : LabelsToDrop_{labelsToDrop}
            , LabelsToAdd_{labelsToAdd}
            , LabelsWithSubs_{labelsWithSubs}
    {
        CalcSizeBytes();
    }

private:
    size_t SizeBytes_ = sizeof(*this);
    THashSet<TLabelBackend> LabelsToDrop_;
    TLabelsHashMap LabelsToAdd_;
    TVector<TLabelNameWithExpr> LabelsWithSubs_;
};

class TMetricTransformer::TRule {
public:
    explicit TRule(const TAggrRuleItem& rule)
        : Selectors_(BuildSelectors(rule.Cond))
        , Transformer_(BuildTransformer(rule.Target))
    {
        CalcSizeBytes();
    }

    bool IsMatch(const TLabelsHashMap& labels) const {
        for (const auto& selector: Selectors_) {
            auto it = labels.find(selector.Name);
            if (it == labels.end() && !selector.Selector.IsAbsent()) {
                return false;
            }
            if (it != labels.end() && !selector.Selector.IsMatch(it->second)) {
                return false;
            }
        }
        return true;
    }

    bool MatchTransform(TLabelsHashMap& labels) {
        if (!IsMatch(labels)) {
            return false;
        }
        Transformer_.Transform(labels);
        return true;
    }

    void CalcSizeBytes() {
        SizeBytes_ = sizeof(*this);
        for (auto& selector: Selectors_) {
            SizeBytes_ += selector.Name.capacity();
            SizeBytes_ += selector.Selector.SizeBytes();
        }
        SizeBytes_ += (Selectors_.capacity() - Selectors_.size()) * sizeof(TLabelSelector);
        SizeBytes_ += Selectors_.capacity() * sizeof(TLabelBackend);
        SizeBytes_ += Transformer_.SizeBytes() - sizeof(Transformer_);
    }

    size_t SizeBytes() const {
        Y_VERIFY(SizeBytes_ < (1l << 34));
        return SizeBytes_;
    }

private:
    struct TLabelSelector {
        TLabelSelector(TStringBuf name, TStringBuf value)
            : Name(name)
            , Selector(TValueSelector::ParseFromStr(value))
        {}

        TLabelBackend Name;
        TValueSelector Selector;
    };

    size_t SizeBytes_ = sizeof(*this);
    TVector<TLabelSelector> Selectors_;
    TLabelsTransformer Transformer_;

private:

    static TVector<TLabelSelector> BuildSelectors(const TVector<TString>& condStrs) {
        TLabels matchLabels;
        for (const auto& condStr: condStrs) {
            auto label = ParseCondition(condStr);
            if (!matchLabels.Add(label)) {
                ythrow yexception()
                        << "multiple actions or matchers were defined with a label \""
                        << label.Name() << "\" inside one Rule";
            }
        }

        TVector<TLabelSelector> Selectors;
        Selectors.reserve(matchLabels.size());
        for (const auto& label: matchLabels) {
            Selectors.emplace_back(label.Name(), label.Value());
        }
        return Selectors;
    }

    static TLabelsTransformer BuildTransformer(const TVector<TString>& targetStrs) {
        TLabels target;
        for (const auto& targetStr: targetStrs) {
            auto label = ParseTarget(targetStr);
            if (!target.Add(label)) {
                ythrow yexception()
                        << "multiple actions or matchers were defined with a label \""
                        << label.Name() << "\" inside one Rule";
            }
        }
        return TLabelsTransformer(std::move(target));
    }

};

TMetricTransformer::TMetricTransformer(const TVector<TAggrRuleItem>& aggrRules) {
    Rules_.reserve(aggrRules.size());
    for (const auto& rule: aggrRules) {
        Rules_.emplace_back(MakeHolder<TRule>(rule));
    }
}

TLabels TMetricTransformer::Transform(const TLabels& labels) {
    if (Rules_.empty()) {
        return {};
    }

    TLabelsHashMap labelsMap;
    for (const auto& label: labels) {
        labelsMap[label.Name()] = label.Value();
    }

    bool anyMatch = false;
    for (const auto& rule: Rules_) {
        anyMatch |= rule->MatchTransform(labelsMap);
    }

    if (!anyMatch) {
        return {};
    }

    TLabels result;
    for (const auto& [name, value]: labelsMap) {
        result.Add(name, value);
    }
    result.SortByName();

    return result;
}

size_t TMetricTransformer::SizeBytes() const {
    size_t res = sizeof(*this);
    res += Rules_.capacity() * sizeof(decltype(Rules_)::value_type);
    for (const auto& rule: Rules_) {
        if (rule) {
            res += rule->SizeBytes();
        }
    }
    Y_VERIFY(res < (1l << 34));
    return res;
}

TMetricTransformer::~TMetricTransformer() {}

} // namespace NSolomon::NIngestor
