#pragma once

#include <solomon/services/ingestor/lib/shard_id.h>

#include <solomon/libs/cpp/sync/rw_lock.h>

#include <library/cpp/monlib/metrics/labels.h>

#include <util/datetime/base.h>

#include <optional>
#include <unordered_map>

namespace NSolomon::NIngestor {

class TPrevValuesSupplier {
public:
    using TLabels = NMonitoring::TLabels;

    struct TValueWithTs {
        double Value;
        TInstant Ts;
    };

public:
    TPrevValuesSupplier(size_t maxSkipsAllowed, size_t deleteMetricsPeriod)
        : MaxSkipsAllowed_(maxSkipsAllowed)
        , DeleteMetricsPeriod_(deleteMetricsPeriod)
        , IntervalsCounter_(0u)
    {
    }

    std::optional<TValueWithTs> GetPrevValueAndUpdate(
            TSourceId sourceId,
            const TLabels& labels,
            double newValue,
            TInstant ts)
    {
        TIntrusivePtr<TRatesTableWithCounter> ratesPtr;
        {
            auto table = Table_.Read();
            if (auto it = table->find(sourceId); it != table->end()) {
                ratesPtr = it->second;
                ratesPtr->LastAccess = IntervalsCounter_;
            }
        }

        if (!ratesPtr) {
            auto newRatesCounter = MakeIntrusive<TRatesTableWithCounter>(IntervalsCounter_);

            auto table = Table_.Write();
            auto [it, _] = table->emplace(sourceId, std::move(newRatesCounter));
            ratesPtr = it->second;
        }

        return ratesPtr->GetPrevValueAndUpdate(labels, newValue, ts);
    }

    void CloseInterval() {
        EraseOldValues();
        ++IntervalsCounter_;
    }

    size_t SizeBytes() const {
        size_t res = sizeof(*this);

        auto table = Table_.Read();
        for (const auto& it: *table) {
            res += sizeof(it.first);
            res += sizeof(it.second);
            res += it.second->SizeBytes();
        }
        Y_VERIFY(res < (1l << 34));
        return res;
    }

private:
    using TIntervalsCounter = ui64;

    struct TRatesTableValue {
        TValueWithTs Value;
        TIntervalsCounter LastAccess;

        TRatesTableValue(TValueWithTs valueWithTs)
            : Value(valueWithTs) {}

        TRatesTableValue(TValueWithTs valueWithTs, TIntervalsCounter lastAccess)
            : Value(valueWithTs)
            , LastAccess(lastAccess) {}
    };

    struct TLabelsHasher {
        ui64 operator()(const TLabels& labels) const {
            return labels.Hash();
        }
    };

    using TRatesTable = std::unordered_map<TLabels, TRatesTableValue, TLabelsHasher>;

    struct TRatesTableWithCounter: public TAtomicRefCount<TRatesTableWithCounter> {
        TRatesTable Rates;
        TIntervalsCounter LastAccess;

        TRatesTableWithCounter(TRatesTable rates, TIntervalsCounter lastAccess)
            : Rates{std::move(rates)}
            , LastAccess{lastAccess}
        {
        }

        explicit TRatesTableWithCounter(TIntervalsCounter lastAccess) noexcept
            : LastAccess{lastAccess}
        {
        }

        size_t SizeBytes() const {
            size_t res = sizeof(*this);
            for (const auto& rate: Rates) {
                for (const auto& label: rate.first) {
                    res += label.Name().size();
                    res += sizeof(label.Name());
                    res += label.Value().size();
                    res += sizeof(label.Value());
                }
                res += sizeof(rate.second);
            }
            Y_VERIFY(res < (1l << 34));
            return res;
        }
        std::optional<TValueWithTs> GetPrevValueAndUpdate(const TLabels& labels, double newValue, TInstant ts) {
            auto iter = Rates.find(labels);
            if (iter == Rates.end()) {
                auto item = std::make_pair(labels, TRatesTableValue{TValueWithTs{newValue, ts}});
                auto [tableIter, _] = Rates.emplace(std::move(item));
                TRatesTableValue& ratesTableValue = tableIter->second;
                ratesTableValue.LastAccess = LastAccess;
                return std::nullopt;
            }

            TRatesTableValue& ratesTableValue = iter->second;
            TValueWithTs result = ratesTableValue.Value;
            ratesTableValue.Value = TValueWithTs{newValue, ts};
            ratesTableValue.LastAccess = LastAccess;

            return result;
        }

        void EraseOldValues(TIntervalsCounter currentInterval, size_t maxSkipsAllowed) {
            auto iter = Rates.begin();
            while (iter != Rates.end()) {
                if (currentInterval - iter->second.LastAccess >= maxSkipsAllowed) {
                    Rates.erase(iter++);
                } else {
                    ++iter;
                }
            }
        }
    };

    void EraseOldValues() {
        auto table = Table_.Write();
        for (auto it = table->begin(), end = table->end(); it != end;) {
            auto ratesTableWithCounter = it->second;
            Y_VERIFY_DEBUG(ratesTableWithCounter->LastAccess <= IntervalsCounter_);
            if (IntervalsCounter_ - ratesTableWithCounter->LastAccess >= MaxSkipsAllowed_) {
                table->erase(it++);
            } else {
                if (IntervalsCounter_ % DeleteMetricsPeriod_ == 0) {
                    ratesTableWithCounter->EraseOldValues(IntervalsCounter_, MaxSkipsAllowed_);
                }
                ++it;
            }
        }
    }

private:
    NSync::TLightRwLock<std::unordered_map<TSourceId, TIntrusivePtr<TRatesTableWithCounter>>> Table_;
    const size_t MaxSkipsAllowed_;
    const size_t DeleteMetricsPeriod_;
    TIntervalsCounter IntervalsCounter_;
};

} // namespace NSolomon::NIngestor
