#pragma once

#include "data_processor.h"

#include <solomon/libs/cpp/log_writer/log_writer.h>
#include <solomon/services/ingestor/lib/shard_config/shard_config.h>
#include <solomon/services/ingestor/lib/shard_id.h>
#include <solomon/libs/cpp/intern/str_pool.h>

#include <library/cpp/monlib/encode/format.h>
#include <library/cpp/monlib/metrics/labels.h>
#include <library/cpp/monlib/metrics/metric_consumer.h>
#include <library/cpp/resource/resource.h>

#include <util/generic/ptr.h>

namespace NSolomon::NIngestor {

class TShard {
public:
    TShard(
            TShardConfig config,
            TLabelPool& labelPool,
            std::shared_ptr<IThreadPool> processorsExecutor,
            std::function<void()> onAllDataProcessed,
            const TString& yasmPrefix)
        : Config_(std::move(config))
        , ShardId_{Config_.ShardStrId}
        , LabelPool_(labelPool)
        , Processor_(
            std::move(processorsExecutor),
            Config_.ShardNumId,
            IsYasmShard(Config_),
            Config_.IsShardMemOnly,
            std::move(config.AggrRules),
            GetYasmAggrRules(Config_, yasmPrefix),
            LabelPool_,
            std::move(onAllDataProcessed))
    {
        Y_ENSURE(!ShardId_.empty(), "empty ShardId value in a config");
    }

    void UpdateConfig(const TShardConfig& config);

    NThreading::TFuture<TDataProcessor::TProcessingResult> ProcessData(
            std::vector<std::unique_ptr<TShardData>> data,
            TInstant intervalStart,
            TDuration intervalLength);

    NThreading::TFuture<TDataProcessor::TProcessingResult> ProcessAggregates(
            TInstant intervalStart,
            TDuration intervalLength);

    size_t SizeBytes() const {
        size_t res = sizeof(*this);
        res += Config_.SizeBytes() - sizeof(Config_);
        res += ShardId_.capacity();
        res += Processor_.SizeBytes() - sizeof(Processor_);
        Y_VERIFY(res < (1l << 34));
        return res;
    }

private:
    TShardConfig Config_;
    const TString& ShardId_;
    TLabelPool& LabelPool_;
    TDataProcessor Processor_;

private:

    static constexpr TStringBuf YASM_PREFIX = "yasm_";
    static constexpr TStringBuf DEFAULT_YASM_PREFIX = "yasm_pull_";
    static constexpr TStringBuf YASM_RULES_CONFIG_PATH = "/rules.conf";

    static bool IsYasmShard(const TShardConfig& config) {
        return config.Key.ProjectId.StartsWith(YASM_PREFIX);
    }

    TVector<TYasmAggrRule> GetYasmAggrRules(const TShardConfig& config, const TString& yasmPrefix) {
        if (!IsYasmShard(config)) {
            return {};
        }

        if (!config.Key.ProjectId.StartsWith(yasmPrefix)) {
            Cerr << "invalid projectId: " << config.Key.ProjectId << " not starts with " << yasmPrefix << '\n';
            return {};
        }

        TString rulesConf = NResource::Find(YASM_RULES_CONFIG_PATH);
        TString project(config.Key.ProjectId.begin() + yasmPrefix.size(), config.Key.ProjectId.end());

        return GetYasmAggrRulesForProject(rulesConf, project, LabelPool_);
    }
};
} // namespace NSolomon::NIngestor
