#include <solomon/services/ingestor/lib/shard/metric_transformer.h>

#include <library/cpp/testing/gtest/gtest.h>

using NMonitoring::TLabels;
using namespace NSolomon::NIngestor;

bool IsLabelsEq(TLabels lhs, TLabels rhs) {
    lhs.SortByName();
    rhs.SortByName();
    return lhs == rhs;
}

TEST(TTransformationTest, Simple) {
    {
        TVector<TAggrRuleItem> rules{};
        TMetricTransformer transformer(rules);
        TLabels labels{{"host", "host"}};

        TLabels actual = transformer.Transform(labels);
        TLabels expected{};

        ASSERT_TRUE(IsLabelsEq(actual, expected));
    }

    {
        TVector<TAggrRuleItem> rules{
            {
                {"host=*"}, {"host=-"}
            }
        };

        TMetricTransformer transformer(rules);
        TLabels labels{{"host", "value"},
                       {"dc",   "sas"}};

        TLabels actual = transformer.Transform(labels);
        TLabels expected{{"dc", "sas"},};

        ASSERT_TRUE(IsLabelsEq(actual, expected));
    }
}

TEST(TTransformationTest, AbsentCondition) {
    TVector<TAggrRuleItem> rules{
        {
            {"label=-"}, {"label=value"}
        }
    };

    TMetricTransformer transformer(rules);

    {
        TLabels labels{{"host", "host.yandex.net"}};

        TLabels actual = transformer.Transform(labels);
        TLabels expected{{"host",  "host.yandex.net"},
                         {"label", "value"}};

        ASSERT_TRUE(IsLabelsEq(actual, expected));
    }

    {
        TLabels labels{{"host", "host.yandex.net"}, {"label", "someValue"}};

        TLabels actual = transformer.Transform(labels);

        ASSERT_TRUE(actual.empty());
    }

}

TEST(TTransformationTest, PositiveGlob) {
    TVector<TAggrRuleItem> rules{
        {
            {"label=glo*"}, {"label=value"}
        }
    };
    TMetricTransformer transformer(rules);

    {
        TLabels labels{{"label", "glob"}};

        TLabels actual = transformer.Transform(labels);
        TLabels expected{{"label", "value"}};

        ASSERT_TRUE(IsLabelsEq(actual, expected));
    }

    {
        TLabels labels{{"label", "glab"}};

        TLabels actual = transformer.Transform(labels);
        TLabels expected{};

        ASSERT_TRUE(IsLabelsEq(actual, expected));
    }
}

TEST(TTransformationTest, NegativeGlob) {
    TVector<TAggrRuleItem> rules{
        {
            {"label=!glo*"}, {"label=value"}
        }
    };
    TMetricTransformer transformer(rules);

    {
        TLabels labels{{"label", "glab"}};

        TLabels actual = transformer.Transform(labels);
        TLabels expected{{"label", "value"}};

        ASSERT_TRUE(IsLabelsEq(actual, expected));
    }

    {
        TLabels labels{{"label", "glob"}};

        TLabels actual = transformer.Transform(labels);
        TLabels expected{};

        ASSERT_TRUE(IsLabelsEq(actual, expected));
    }
}

TEST(TTransformationTest, SeveralPatternsInCondition) {
    TVector<TAggrRuleItem> rules{
        {
            {"label=*|-"}, {"label=value"}
        }
    };
    TMetricTransformer transformer(rules);

    {
        TLabels labels{{"host", "host.yandex.net"}};

        TLabels actual = transformer.Transform(labels);
        TLabels expected{{"host",  "host.yandex.net"},
                         {"label", "value"}};

        ASSERT_TRUE(IsLabelsEq(actual, expected));
    }

    {
        TLabels labels{{"host", "host.yandex.net"}, {"label", "someValue"}};

        TLabels actual = transformer.Transform(labels);
        TLabels expected{{"host",  "host.yandex.net"},
                         {"label", "value"}};

        ASSERT_TRUE(IsLabelsEq(actual, expected));
    }
}

TEST(TTransformationTest, SeveralPatternsInCondition2) {
    TVector<TAggrRuleItem> rules{
        {
            {"label=value1|value2"}, {"label=someValue"}
        }
    };
    TMetricTransformer transformer(rules);

    {
        TLabels labels{{"label", "value1"},
                       {"dc",    "vla"}};
        TLabels actual = transformer.Transform(labels);
        TLabels expected{{"label", "someValue"},
                         {"dc",   "vla"}};

        ASSERT_TRUE(IsLabelsEq(actual, expected));
    }

    {
        TLabels labels{{"label", "value3"},
                       {"dc",    "vla"}};
        TLabels actual = transformer.Transform(labels);
        TLabels expected{};

        ASSERT_TRUE(IsLabelsEq(actual, expected));
    }
}

TEST(TTransformationTest, Substitutions) {
    TVector<TAggrRuleItem> rules{
        {
            {"label=*"}, {"host=-", "label={{host}}"}
        }
    };
    TMetricTransformer transformer(rules);
    TLabels labels{{"label", "value"}, {"host", "host.yandex.net"}};

    TLabels actual = transformer.Transform(labels);
    TLabels expected{{"label", "host.yandex.net"}};

    ASSERT_TRUE(IsLabelsEq(actual, expected));
}

TEST(TTransformationTest, SubstitutionsOrder) {
    TVector<TAggrRuleItem> rules{
        {
            {"label=*"}, {"host=newHost", "label={{host}}"}
        }
    };
    TMetricTransformer transformer(rules);
    TLabels labels{{"label", "value"}, {"host", "host.yandex.net"}};

    TLabels actual = transformer.Transform(labels);
    TLabels expected{{"label", "host.yandex.net"}, {"host", "newHost"}};

    ASSERT_TRUE(IsLabelsEq(actual, expected));
}

TEST(TTransformationTest, ManyNegativePatterns) {
    TVector<TAggrRuleItem> rules{
        {
            {"label=!value1|value2"}, {"label=value"}
        }
    };
    TMetricTransformer transformer(rules);

    TLabels labels1{{"label", "value1"}, {"host", "host.yandex.net"}};
    TLabels labels2{{"label", "value2"}, {"host", "host.yandex.net"}};
    TLabels labels3{{"label", "value3"}, {"host", "host.yandex.net"}};

    TLabels actual1 = transformer.Transform(labels1);
    TLabels actual2 = transformer.Transform(labels2);
    TLabels actual3 = transformer.Transform(labels3);

    TLabels expected1{};
    TLabels expected2{};
    TLabels expected3{{"label", "value"}, {"host", "host.yandex.net"}};

    ASSERT_TRUE(IsLabelsEq(actual1, expected1));
    ASSERT_TRUE(IsLabelsEq(actual2, expected2));
    ASSERT_TRUE(IsLabelsEq(actual3, expected3));
}

TEST(TTransformationTest, ManyRulesManyPatterns) {
    TVector<TAggrRuleItem> rules{
        {
            {"label=*", "label1=val*|smth", "label2=-", "label3=!value3"},
            {"label=-", "label1=value", "label2={{label}}"}
        },
        {
            {"host=*", "dc=sas"},
            {"dc=-", "host={{dc}}"}
        }
    };
    TMetricTransformer transformer(rules);

    {
        // match first rule
        TLabels labels{
            {"label",  "value"},
            {"label1", "value"},
            {"label3", "someValue"},
            {"host", "host.yandex.net"},
            {"dc", "vla"}
        };

        TLabels actual = transformer.Transform(labels);
        TLabels expected{
            {"label1", "value"},
            {"label2", "value"},
            {"label3", "someValue"},
            {"host", "host.yandex.net"},
            {"dc", "vla"}
        };

        ASSERT_TRUE(IsLabelsEq(actual, expected));
    }

    {
        // match second rule
        TLabels labels{
            {"label",  "value"},
            {"label1", "value"},
            {"label3", "value3"},
            {"host", "host.yandex.net"},
            {"dc", "sas"}
        };

        TLabels actual = transformer.Transform(labels);
        TLabels expected{
            {"label",  "value"},
            {"label1", "value"},
            {"label3", "value3"},
            {"host", "sas"},
        };

        ASSERT_TRUE(IsLabelsEq(actual, expected));
    }

    {
        // match both rules
        TLabels labels{
            {"label",  "someValue"},
            {"label1", "smth"},
            {"label3", "value"},
            {"host", "host.yandex.net"},
            {"dc", "sas"}
        };

        TLabels actual = transformer.Transform(labels);
        TLabels expected{
            {"label2", "someValue"},
            {"label1", "value"},
            {"label3", "value"},
            {"host", "sas"},
        };

        ASSERT_TRUE(IsLabelsEq(actual, expected));
    }

    {
        // don't match any rules
        TLabels labels1{
            {"dc", "sas"},
            // has no "host" label, don't match 2 rule

            {"label", "smth"},
            {"label2", "smth"},
            {"label1", "smth"},
            {"label3", "smth"}
            // has "label2" label, so don't match 1 rule
        };

        TLabels actual1 = transformer.Transform(labels1);
        TLabels expected{};

        ASSERT_TRUE(IsLabelsEq(actual1, expected));
    }
}

TEST(TTransformationTest, InvalidRules) {
//    {
//        TVector<TAggrRuleItem> rules{
//            {
//                {"cluster=*", "service=*"},
//                {"cluster=-", "cluster=123"}
//            }
//        };
//
//        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
//                TMetricTransformer{rules},
//                yexception,
//                "cannot drop and add a value at the same time");
//    }

//    {
//        TVector<TAggrRuleItem> rules{
//            {
//                {"cluster=*", "service=*"},
//                {"cluster=123", "cluster={{someValue}}"}
//            }
//        };
//
//        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
//                TMetricTransformer{rules},
//                yexception,
//                "cannot perform multiple actions on one label inside one Rule");
//    }

    {
        TVector<TAggrRuleItem> rules{
            {
                {"cluster=*", "cluster=123"},
                {"newName=newValue"}
            }
        };


        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                TMetricTransformer{rules},
                yexception,
                "multiple actions or matchers were defined with a label \"cluster\" inside one Rule");
    }

    {
        TVector<TAggrRuleItem> rules{
            {
                {"cluster=*", "service=123"},
                {"newName="}
            }
        };


        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                TMetricTransformer{rules},
                yexception,
                "Label value(=) in a ReplaceMeta is empty or has non-printable characters");
    }

    {
        TVector<TAggrRuleItem> rules1{
            {
                {"cluster="},
                {"newName=123"}
            }
        };
        TVector<TAggrRuleItem> rules2{
            {
                {"cluster=!"},
                {"newName=123"}
            }
        };
        TVector<TAggrRuleItem> rules3{
            {
                {"cluster=*|"},
                {"newName=123"}
            }
        };

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                TMetricTransformer{rules1},
                yexception,
                "Label value(=) in a Match is empty or has non-printable characters");

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                TMetricTransformer{rules2},
                yexception,
                "empty condition");

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                TMetricTransformer{rules3},
                yexception,
                "empty condition");
    }

    {
        // is this match rule invalid?
        TVector<TAggrRuleItem> rules{
            {
                {"cluster=!-"},
                {"newName=123"}
            }
        };

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                TMetricTransformer{rules},
                yexception,
                "invalid condition"); // is it?
    }
}
