#include <solomon/services/ingestor/lib/shard/prev_values_supplier.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon::NIngestor;

class TPrevValuesTest: public ::testing::Test {
public:
    const NMonitoring::TLabels Labels{ {"name", "value"} };
    const TInstant Ts = TInstant::MilliSeconds(0);
    const TDuration Length = TDuration::Seconds(15);
};

TEST_F(TPrevValuesTest, Simple) {
    TPrevValuesSupplier prevValues(1u, 1u);

    auto value = prevValues.GetPrevValueAndUpdate(0, Labels, 0.0, Ts);
    ASSERT_FALSE(value.has_value());

    prevValues.CloseInterval();

    value = prevValues.GetPrevValueAndUpdate(0, Labels, 1.0, Ts + Length);
    ASSERT_TRUE(value.has_value());
    ASSERT_DOUBLE_EQ(value.value().Value, 0.0);
    ASSERT_EQ(value.value().Ts, Ts);
}

TEST_F(TPrevValuesTest, SeqUpdates) {
    TPrevValuesSupplier prevValues(1u, 1u);
    const size_t attempts = 100;

    TInstant ts = Ts;
    prevValues.GetPrevValueAndUpdate(0, Labels, double(0.0), ts);

    for (size_t i = 1; i < attempts; ++i) {
        prevValues.CloseInterval();
        ts += Length;
        auto value = prevValues.GetPrevValueAndUpdate(0, Labels, double(i), ts);
        ASSERT_TRUE(value.has_value());
        ASSERT_DOUBLE_EQ(value.value().Value, double(i - 1));
        ASSERT_EQ(value.value().Ts, ts - Length);
    }
}

TEST_F(TPrevValuesTest, SourceIdDeletes) {
    const size_t maxSkipsAllowed = 5;
    const size_t deleteMetricsPeriod = 5;

    TPrevValuesSupplier prevValues(maxSkipsAllowed, deleteMetricsPeriod);
    prevValues.GetPrevValueAndUpdate(0, Labels, 0.0, Ts);

    for (size_t i = 0; i < maxSkipsAllowed; ++i) {
        prevValues.CloseInterval();
    }

    auto value = prevValues.GetPrevValueAndUpdate(0, Labels, 1.0, Ts + Length);
    ASSERT_TRUE(value.has_value());

    for (size_t i = 0; i < maxSkipsAllowed + 1; ++i) {
        prevValues.CloseInterval();
    }

    value = prevValues.GetPrevValueAndUpdate(0, Labels, 2.0, Ts + Length + Length);
    ASSERT_FALSE(value.has_value());
}

TEST_F(TPrevValuesTest, LabelsDeletes) {
    const size_t maxSkipsAllowed = 5;
        const size_t deleteMetricsPeriod = 5;
    const NMonitoring::TLabels Labels2{ {"name2", "value2"} };
    auto ts = Ts;

    TPrevValuesSupplier prevValues(maxSkipsAllowed, deleteMetricsPeriod);
    prevValues.GetPrevValueAndUpdate(0, Labels, 0.0, Ts);
    prevValues.GetPrevValueAndUpdate(0, Labels2, 0.0, Ts);

    prevValues.CloseInterval();
    ts += Length;
    prevValues.GetPrevValueAndUpdate(0, Labels2, 1.0, ts);

    for (size_t i = 0; i < maxSkipsAllowed; ++i) {
        prevValues.CloseInterval();
    }

    auto value = prevValues.GetPrevValueAndUpdate(0, Labels, 1.0, ts + Length);
    ASSERT_FALSE(value.has_value());
    value = prevValues.GetPrevValueAndUpdate(0, Labels2, 2.0, ts + Length);
    ASSERT_TRUE(value.has_value());
    ASSERT_DOUBLE_EQ(value.value().Value, 1.0);
}
