#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/services/ingestor/lib/shard/data_processor.h>

#include <solomon/libs/cpp/slog/slog.h>
#include <solomon/libs/cpp/slog/unresolved_meta/iterator.h>
#include <solomon/libs/cpp/yasm/constants/labels.h>

#include <library/cpp/monlib/encode/spack/spack_v1.h>
#include <library/cpp/monlib/metrics/labels.h>

#include <util/generic/string.h>

using namespace NSolomon;
using namespace NSolomon::NIngestor;
using namespace NMonitoring;

using yandex::solomon::common::UrlStatusType;

const TInstant INSTANT = TInstant::ParseIso8601("2021-01-00T00:00:00Z");
constexpr TDuration INTERVAL = TDuration::Seconds(5);
constexpr ui32 SHARD_ID = 1;

TString VALID_JSON = R"({
        "sensors": [
            {
                "labels": {
                    "signal": "count_summ"
                },
                "value": 1
            },
            {
                "labels": {
                    "signal": "count2_summ"
                },
                "value": 2
            }
        ]
    })";

auto TestSpackEncoder(IOutputStream* s) {
    return EncoderSpackV1(
            s,
            ETimePrecision::MILLIS,
            ECompression::IDENTITY
    );
}

using TMetrics = std::unordered_map<NMonitoring::TLabels, NSolomon::TAggrTimeSeries, THash<NMonitoring::TLabels>>;

void CompareSummary(ISummaryDoubleSnapshot* lhs, ISummaryDoubleSnapshot* rhs) {
    EXPECT_EQ(lhs->GetSum(), rhs->GetSum());
    EXPECT_EQ(lhs->GetMin(), rhs->GetMin());
    EXPECT_EQ(lhs->GetMax(), rhs->GetMax());
    EXPECT_EQ(lhs->GetLast(), rhs->GetLast());
    EXPECT_EQ(lhs->GetCount(), rhs->GetCount());
}

void CompareSeries(const NSolomon::TAggrTimeSeries& actual, const NSolomon::TAggrTimeSeries& expected) {
    ASSERT_EQ(actual.Size(), expected.Size());
    ASSERT_EQ(actual.GetValueType(), expected.GetValueType());

    for (size_t i = 0; i < actual.Size(); ++i) {
        EXPECT_EQ(actual[i].GetTime(), expected[i].GetTime());
        EXPECT_EQ(actual[i].GetCount(), expected[i].GetCount());
        switch (actual.GetValueType()) {
            case EMetricValueType::SUMMARY: {
                CompareSummary(actual[i].GetValue().AsSummaryDouble(), expected[i].GetValue().AsSummaryDouble());
                break;
            }

            default: {
                ASSERT_TRUE(false);
            }
        }
    }
}

void Verify(const TMetrics& expected, const TString& d, const TString& m) {
    TStringStream dataStream(d);
    TStringStream metaStream(m);

    TMetrics actual;

    auto dataIt = NSolomon::NSlog::CreateLogDataIterator(&dataStream);
    auto metaIt = NSolomon::NSlog::NUnresolvedMeta::CreateUnresolvedMetaIterator(&metaStream);

    while (dataIt->HasNext()) {
        ASSERT_TRUE(metaIt->HasNext());

        auto data = dataIt->Next();
        auto meta = metaIt->Next();

        NMonitoring::TLabels monLa;
        for (const auto& l: meta.Labels) {
            monLa.Add(l.first, l.second);
        }

        monLa.SortByName();

        if (actual.find(monLa) == actual.end()) {
            actual[monLa] = std::move(data.TimeSeries);
        } else {
            auto& series = actual[monLa];
            series.Extend(std::move(data.TimeSeries));
            series.SortByTs();
        }
    }

    EXPECT_EQ(expected.size(), actual.size());
    for (const auto& [key, value]: actual) {
        ASSERT_TRUE(expected.find(key) != expected.end()) << TStringBuf(TStringBuilder() << key) << " not found";

        CompareSeries(value, expected.at(key));
    }
}

TString CreateGauge(const NMonitoring::TLabels& labels, double value) {
    TString res;
    TStringOutput s(res);
    auto encoder = TestSpackEncoder(&s);

    encoder->OnStreamBegin();

    encoder->OnMetricBegin(EMetricType::GAUGE);

    encoder->OnLabelsBegin();

    for (const auto& label: labels) {
        encoder->OnLabel(label.Name(), label.Value());
    }

    encoder->OnLabelsEnd();

    encoder->OnDouble(TInstant::Zero(), value);

    encoder->OnMetricEnd();

    encoder->OnStreamEnd();
    encoder->Close();

    return res;
}

std::shared_ptr<IThreadPool> TestExecutor() {
    return std::shared_ptr<IThreadPool>(CreateThreadPool(1).Release());
}

TDataProcessor TestDataProcessor(TVector<TYasmAggrRule> aggrRules, TLabelPool& labelPool) {
    return TDataProcessor(TestExecutor(), SHARD_ID, true, false, {}, std::move(aggrRules), labelPool, [](){});
}

TEST(TDataProcessor, JsonError) {
    TLabelPool pool;
    auto processor = TestDataProcessor({}, pool);

    TString nonValidJson = R"({
        "sensors": [{
            "labels": {
                "key": "val)";

    TShardData shardData;
    shardData.Host = "localhost";
    shardData.TimesMillis = INSTANT;
    shardData.Data = nonValidJson;
    shardData.MetricsFormat = EFormat::JSON;

    std::vector<std::unique_ptr<TShardData>> batch;
    batch.emplace_back(std::make_unique<TShardData>(std::move(shardData)));

    auto f = processor.ProcessBatch(std::move(batch), INSTANT, INTERVAL);
    auto res = f.ExtractValueSync();

    ASSERT_EQ(res.Statuses.size(), 1u);
    auto& status = res.Statuses.front();
    EXPECT_EQ(status.MetricsWritten, 0u);
    EXPECT_EQ(status.StatusType, UrlStatusType::JSON_ERROR);
}

TEST(TDataProcessor, SpackError) {
    TLabelPool pool;
    auto processor = TestDataProcessor({}, pool);

    TShardData shardData;
    shardData.Host = "localhost";
    shardData.TimesMillis = INSTANT;
    shardData.MetricsFormat = EFormat::SPACK;

    std::vector<std::unique_ptr<TShardData>> batch;
    batch.emplace_back(std::make_unique<TShardData>(std::move(shardData)));

    auto f = processor.ProcessBatch(std::move(batch), INSTANT, INTERVAL);
    auto res = f.ExtractValueSync();

    ASSERT_EQ(res.Statuses.size(), 1u);
    auto& status = res.Statuses.front();
    EXPECT_EQ(status.MetricsWritten, 0u);
    EXPECT_EQ(status.StatusType, UrlStatusType::SPACK_ERROR);
}

TEST(TDataProcessor, QuotaError) {
    TLabelPool pool;
    auto processor = TestDataProcessor({}, pool);

    TMetricProcessorOptions opts{
        .ResponseTs = INSTANT,
        .Quota = {
            .MaxMetricsPerUrl = 1
        },
        .IntervalStart = INSTANT,
        .IntervalLength = INTERVAL,
        .LabelPool = &pool,
    };

    auto f = processor.GetMetaAndData(VALID_JSON, opts, NMonitoring::EFormat::JSON);
    auto res = f.ExtractValueSync();

    ASSERT_EQ(res.Statuses.size(), 1u);
    auto& status = res.Statuses.front();
    EXPECT_EQ(status.MetricsWritten, 0u);
    EXPECT_EQ(status.StatusType, UrlStatusType::QUOTA_ERROR);
}

TEST(TDataProcessor, ParseError) {
    TLabelPool pool;
    auto processor = TestDataProcessor({}, pool);

    {
        TShardData sd;
        sd.TimesMillis = INSTANT;
        sd.Data = VALID_JSON;
        sd.MetricsFormat = EFormat::UNKNOWN;

        std::vector<std::unique_ptr<TShardData>> batch;
        batch.emplace_back(std::make_unique<TShardData>(std::move(sd)));

        auto f = processor.ProcessBatch(std::move(batch), INSTANT, INTERVAL);
        auto res = f.ExtractValueSync();

        ASSERT_EQ(res.Statuses.size(), 1u);
        auto& status = res.Statuses.front();
        EXPECT_EQ(status.MetricsWritten, 0u);
        ASSERT_EQ(status.StatusType, UrlStatusType::PARSE_ERROR);
        EXPECT_TRUE(status.Message.Contains(TStringBuf("unsupported format")));
    }

    {
        TString validJsonButNotMetrics = R"({
            "sensors": [
                {
                    "labels": {
                        "key": "too_small_summ"
                    },
                    "ts": 1,
                    "value": 1
                }
            ]
        })";

        TShardData sd;
        sd.TimesMillis = INSTANT;
        sd.Data = validJsonButNotMetrics;
        sd.MetricsFormat = EFormat::JSON;

        std::vector<std::unique_ptr<TShardData>> batch;
        batch.emplace_back(std::make_unique<TShardData>(std::move(sd)));

        auto f = processor.ProcessBatch(std::move(batch), INSTANT, INTERVAL);
        auto res = f.ExtractValueSync();

        ASSERT_EQ(res.Statuses.size(), 1u);
        auto& status = res.Statuses.front();
        EXPECT_EQ(status.MetricsWritten, 0u);
        ASSERT_EQ(status.StatusType, UrlStatusType::PARSE_ERROR);
        EXPECT_TRUE(status.Message.Contains(TStringBuf("invalid timestamp")));
    }
}

TEST(TDataProcessor, InternalError) {
    TLabelPool pool;
    auto processor = TestDataProcessor({}, pool);

    TMetricProcessorOptions opts;
    opts.ValidationMode = TValidationMode::End; // unknown value, will throw an exception
    opts.ResponseTs = INSTANT;
    opts.LabelPool = &pool;

    auto f = processor.GetMetaAndData(VALID_JSON, opts, NMonitoring::EFormat::JSON);
    auto res = f.ExtractValueSync();

    ASSERT_EQ(res.Statuses.size(), 1u);
    auto& status = res.Statuses.front();
    EXPECT_EQ(status.MetricsWritten, 0u);
    ASSERT_EQ(status.StatusType, UrlStatusType::UNKNOWN_ERROR);
}

TEST(TDataProcessor, HostValidMetric) {
    TLabelPool pool;
    TVector<TYasmAggrRule> rules{
        TYasmAggrRule{pool.Intern("ctype"), pool.Intern("geo")},
        TYasmAggrRule{pool.Intern("geo")}
    };

    auto processor = TestDataProcessor(rules, pool);

    TShardData shardDataF;

    shardDataF.Host = "localhost";
    shardDataF.TimesMillis = INSTANT;
    shardDataF.Data = CreateGauge(NMonitoring::TLabels{{"signal", "count_summ"}, {"ctype", "none"}, {"geo", "msk"}},  1.0);
    shardDataF.HostOptLabels = NMonitoring::TLabels{},
    shardDataF.MetricsFormat = EFormat::SPACK;

    TShardData shardDataS;

    shardDataS.Host = "somehost";
    shardDataS.TimesMillis = INSTANT + INTERVAL;
    shardDataS.Data = CreateGauge(NMonitoring::TLabels{{"geo", "msk"}, {"yasm_container", "localhost"}, {"ctype", "none"}, {"signal", "count_summ"}}, 2.0);
    shardDataS.HostOptLabels = NMonitoring::TLabels{},
    shardDataS.MetricsFormat = EFormat::SPACK;

    std::vector<std::unique_ptr<TShardData>> batch;

    batch.emplace_back(std::make_unique<TShardData>(std::move(shardDataF)));
    batch.emplace_back(std::make_unique<TShardData>(std::move(shardDataS)));

    auto f = processor.ProcessBatch(std::move(batch), INSTANT, INTERVAL);
    TDataProcessor::TProcessingResult res = f.ExtractValueSync();

    NSolomon::TAggrTimeSeries expectedSeries;
    expectedSeries.Add(INSTANT, MakeIntrusive<TSummaryDoubleSnapshot>(1.0, 0.0, 0.0, 0.0, 0u).Get());
    expectedSeries.Add(INSTANT + INTERVAL, MakeIntrusive<TSummaryDoubleSnapshot>(2.0, 0.0, 0.0, 0.0, 0u).Get());
    TMetrics expected;
    expected[NMonitoring::TLabels{
        {"ctype", "none"},
        {"geo", "msk"},
        {"host", "localhost"},
        {"signal", "count_summ"}}] = std::move(expectedSeries);

    expectedSeries = {};
    expectedSeries.Add(INSTANT + INTERVAL, MakeIntrusive<TSummaryDoubleSnapshot>(2.0, 0.0, 0.0, 0.0, 0u).Get());
    expected[NMonitoring::TLabels{
            {"ctype", NYasm::AGGREGATED_MARKER},
            {"geo", NYasm::AGGREGATED_MARKER},
            {"host", "localhost"},
            {"signal", "count_summ"}}] = std::move(expectedSeries);

    expectedSeries = {};
    expectedSeries.Add(INSTANT + INTERVAL, MakeIntrusive<TSummaryDoubleSnapshot>(2.0, 0.0, 0.0, 0.0, 0u).Get());
    expected[NMonitoring::TLabels{
            {"ctype", "none"},
            {"geo", NYasm::AGGREGATED_MARKER},
            {"host", "localhost"},
            {"signal", "count_summ"}}] = std::move(expectedSeries);

    Verify(expected, res.Data, res.Meta);

    expected.clear();
    expectedSeries = {};
    expectedSeries.Add(INSTANT, MakeIntrusive<TSummaryDoubleSnapshot>(1.0, 0.0, 0.0, 0.0, 0u).Get(), 0, 1);
    expected[NMonitoring::TLabels{
            {"ctype", NYasm::AGGREGATED_MARKER},
            {"geo", NYasm::AGGREGATED_MARKER},
            {"host", "localhost"},
            {"signal", "count_summ"}}] = std::move(expectedSeries);

    expectedSeries = {};
    expectedSeries.Add(INSTANT, MakeIntrusive<TSummaryDoubleSnapshot>(1.0, 0.0, 0.0, 0.0, 0u).Get(), 0, 1);
    expected[NMonitoring::TLabels{
            {"ctype", "none"},
            {"geo", NYasm::AGGREGATED_MARKER},
            {"host", "localhost"},
            {"signal", "count_summ"}}] = std::move(expectedSeries);

    f = processor.ProcessAggregates(INSTANT, INTERVAL);
    res = f.ExtractValueSync();
    Verify(expected, res.Data, res.Meta);
}

TEST(TDataProcessor, GroupValidMetric) {
    TLabelPool pool;
    TVector<TYasmAggrRule> rules{
            TYasmAggrRule{pool.Intern("ctype"), pool.Intern("geo")},
            TYasmAggrRule{pool.Intern("geo")}
    };

    auto processor = TestDataProcessor(rules, pool);

    TShardData data[4];
    data[0].Host = "localhost";
    data[0].TimesMillis = INSTANT;
    data[0].Data = CreateGauge(NMonitoring::TLabels{{"signal", "count_summ"}, {"ctype", "none"}, {"geo", "msk"}}, 1.0);
    data[0].HostOptLabels = NMonitoring::TLabels{ {"group_SAS", "SAS"}, {"group_VLA", "VLA"}};
    data[0].MetricsFormat = EFormat::SPACK;

    data[1].Host = "somehost";
    data[1].TimesMillis = INSTANT;
    data[1].Data = CreateGauge(NMonitoring::TLabels{{"signal", "count_summ"}, {"yasm_container", "somehost"}, {"geo", "msk"}, {"ctype", "none"}}, 3.0);
    data[1].HostOptLabels = NMonitoring::TLabels{ {"group_SAS", "SAS"}, {"group_VLA", "VLA"}};
    data[1].MetricsFormat = EFormat::SPACK;

    data[2].Host = "somehost";
    data[2].TimesMillis = INSTANT;
    data[2].Data = CreateGauge(NMonitoring::TLabels{{"signal", "count_summ"}, {"geo", "msk"}, {"ctype", "none"}}, 5.0);
    data[2].HostOptLabels = NMonitoring::TLabels{ {"group_SAS", "SAS"}, {"group_VLA", "VLA"}};
    data[2].MetricsFormat = EFormat::SPACK;

    data[3].Host = "somehost";
    data[3].TimesMillis = INSTANT + INTERVAL;
    data[3].Data = CreateGauge(NMonitoring::TLabels{{"signal", "count_summ"}, {"geo", "msk"}, {"ctype", "none"}}, 10.0);
    data[3].HostOptLabels = NMonitoring::TLabels{ {"group_SAS", "SAS"}, {"group_VLA", "VLA"}};
    data[3].MetricsFormat = EFormat::SPACK;

    std::vector<std::unique_ptr<TShardData>> batch;
    for (size_t i = 0; i < 4; ++i) {
        batch.emplace_back(std::make_unique<TShardData>(std::move(data[i])));
    }

    auto f = processor.ProcessBatch(std::move(batch), INSTANT, INTERVAL);
    TDataProcessor::TProcessingResult res = f.ExtractValueSync();

    NSolomon::TAggrTimeSeries expectedSeries;
    TMetrics expected;

    NMonitoring::TLabels l1{
            {"ctype", "none"},
            {"geo", "msk"},
            {"group", "SAS"},
            {"host", NYasm::AGGREGATED_MARKER},
            {"signal", "count_summ"}};
    NMonitoring::TLabels l2{
            {"ctype", NYasm::AGGREGATED_MARKER},
            {"geo", NYasm::AGGREGATED_MARKER},
            {"group", "SAS"},
            {"host", NYasm::AGGREGATED_MARKER},
            {"signal", "count_summ"}};
    NMonitoring::TLabels l3{
            {"ctype", NYasm::AGGREGATED_MARKER},
            {"geo", NYasm::AGGREGATED_MARKER},
            {"group", "VLA"},
            {"host", NYasm::AGGREGATED_MARKER},
            {"signal", "count_summ"}};
    NMonitoring::TLabels l4{
            {"ctype", "none"},
            {"geo", NYasm::AGGREGATED_MARKER},
            {"group", "SAS"},
            {"host", NYasm::AGGREGATED_MARKER},
            {"signal", "count_summ"}};
    NMonitoring::TLabels l5{
            {"ctype", "none"},
            {"geo", "msk"},
            {"group", "VLA"},
            {"host", NYasm::AGGREGATED_MARKER},
            {"signal", "count_summ"}};
    NMonitoring::TLabels l6{
            {"ctype", "none"},
            {"geo", NYasm::AGGREGATED_MARKER},
            {"group", "VLA"},
            {"host", NYasm::AGGREGATED_MARKER},
            {"signal", "count_summ"}};

    expectedSeries.Add(INSTANT + INTERVAL, MakeIntrusive<TSummaryDoubleSnapshot>(10.0, 0.0, 0.0, 0.0, 0u).Get(), 0, 1);
    expected[l1] = std::move(expectedSeries);

    expectedSeries = {};
    expectedSeries.Add(INSTANT + INTERVAL, MakeIntrusive<TSummaryDoubleSnapshot>(10.0, 0.0, 0.0, 0.0, 0u).Get(), 0, 1);
    expected[l5] = std::move(expectedSeries);

    expectedSeries = {};
    expectedSeries.Add(INSTANT + INTERVAL, MakeIntrusive<TSummaryDoubleSnapshot>(10.0, 0.0, 0.0, 0.0, 0u).Get(), 0, 1);
    expected[l2] = std::move(expectedSeries);

    expectedSeries = {};
    expectedSeries.Add(INSTANT + INTERVAL, MakeIntrusive<TSummaryDoubleSnapshot>(10.0, 0.0, 0.0, 0.0, 0u).Get(), 0, 1);
    expected[l3] = std::move(expectedSeries);

    expectedSeries = {};
    expectedSeries.Add(INSTANT + INTERVAL, MakeIntrusive<TSummaryDoubleSnapshot>(10.0, 0.0, 0.0, 0.0, 0u).Get(), 0, 1);
    expected[l4] = std::move(expectedSeries);

    expectedSeries = {};
    expectedSeries.Add(INSTANT + INTERVAL, MakeIntrusive<TSummaryDoubleSnapshot>(10.0, 0.0, 0.0, 0.0, 0u).Get(), 0, 1);
    expected[l6] = std::move(expectedSeries);

    Verify(expected, res.Data, res.Meta);

    expected.clear();
    expectedSeries = {};
    expectedSeries.Add(INSTANT, MakeIntrusive<TSummaryDoubleSnapshot>(9.0, 0.0, 0.0, 0.0, 0u).Get(), 0, 3);
    expected[l1] = std::move(expectedSeries);

    expectedSeries = {};
    expectedSeries.Add(INSTANT, MakeIntrusive<TSummaryDoubleSnapshot>(9.0, 0.0, 0.0, 0.0, 0u).Get(), 0, 3);
    expected[l5] = std::move(expectedSeries);

    expectedSeries = {};
    expectedSeries.Add(INSTANT, MakeIntrusive<TSummaryDoubleSnapshot>(9.0, 0.0, 0.0, 0.0, 0u).Get(), 0, 3);
    expected[l2] = std::move(expectedSeries);

    expectedSeries = {};
    expectedSeries.Add(INSTANT, MakeIntrusive<TSummaryDoubleSnapshot>(9.0, 0.0, 0.0, 0.0, 0u).Get(), 0, 3);
    expected[l3] = std::move(expectedSeries);

    expectedSeries = {};
    expectedSeries.Add(INSTANT, MakeIntrusive<TSummaryDoubleSnapshot>(9.0, 0.0, 0.0, 0.0, 0u).Get(), 0, 3);
    expected[l4] = std::move(expectedSeries);

    expectedSeries = {};
    expectedSeries.Add(INSTANT, MakeIntrusive<TSummaryDoubleSnapshot>(9.0, 0.0, 0.0, 0.0, 0u).Get(), 0, 3);
    expected[l6] = std::move(expectedSeries);

    f = processor.ProcessAggregates(INSTANT, INTERVAL);
    res = f.ExtractValueSync();

    Verify(expected, res.Data, res.Meta);
}
