#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/services/ingestor/lib/shard/yasm_metric_transformer.h>

#include <solomon/libs/cpp/yasm/constants/labels.h>

#include <library/cpp/monlib/metrics/labels.h>

namespace NSolomon::NIngestor {


void AssertLabelsEqual(const TLabelsSet& lhs, const TVector<TLabels>& rhs) {
    ASSERT_EQ(lhs.size(), rhs.size());
    TLabelsSet set;
    for (const auto& val: rhs) {
        set.emplace(val);
    }

    ASSERT_EQ(set, lhs);
}

TEST(GetRules, parsing) {
    TString rulsConf = NResource::Find("/rules1.conf");

    {
        TString project("balancer");
        TLabelPool pool;

        TVector<TYasmAggrRule> yasmRules = GetYasmAggrRulesForProject(rulsConf, project, pool);
        ASSERT_EQ(yasmRules.size(), 2u);

        auto expectedRule1 = TYasmAggrRule{pool.Intern("ctype"), pool.Intern("geo"), pool.Intern("prj"), pool.Intern("tier")};
        auto expectedRule2 = TYasmAggrRule{pool.Intern("prj")};
        auto expectedRules = TVector<TYasmAggrRule>{expectedRule1, expectedRule2};

        for (auto rule: expectedRules) {
            auto iter = std::find(expectedRules.begin(), expectedRules.end(), rule);
            ASSERT_TRUE(iter != expectedRules.end());
        }
    }

    {
        TString project("solomon");
        TLabelPool pool;
        TVector<TYasmAggrRule> yasmRules = GetYasmAggrRulesForProject(rulsConf, project, pool);

        ASSERT_EQ(yasmRules.size(), 1u);

        auto expectedRule1 = TYasmAggrRule{pool.Intern("prj")};
        auto expectedRules = TVector<TYasmAggrRule>{expectedRule1};

        for (auto rule: expectedRules) {
            auto iter = std::find(expectedRules.begin(), expectedRules.end(), rule);
            ASSERT_TRUE(iter != expectedRules.end());
        }
    }
}

TEST(Transformer, transform) {
    TLabelPool pool;
    TYasmMetricTransformer transformer({ {pool.Intern("ctype"), pool.Intern("prj"), pool.Intern("geo"), pool.Intern("tier")}, {pool.Intern("prj")} });

    TLabels origin(&pool);
    origin.Add(TStringBuf("ctype"), TStringBuf("a"));
    origin.Add(TStringBuf("geo"), TStringBuf("b"));
    origin.Add(TStringBuf("prj"), TStringBuf("c"));
    origin.Add(TStringBuf("tier"), TStringBuf("d"));

    TVector<TLabels> expected;
    expected.emplace_back(&pool);
    expected.emplace_back(&pool);

    for (const auto& label: NMonitoring::TLabels{{"ctype", NYasm::AGGREGATED_MARKER},
                                                 {"geo", NYasm::AGGREGATED_MARKER},
                                                 {"prj", NYasm::AGGREGATED_MARKER},
                                                 {"tier", NYasm::AGGREGATED_MARKER}}) {
        expected[0].Add(label.Name(), label.Value());
    }

    std::sort(expected[0].begin(), expected[0].end());

    for (const auto& label: NMonitoring::TLabels{{"ctype", "a"},
                                                 {"geo", "b"},
                                                 {"prj", NYasm::AGGREGATED_MARKER},
                                                 {"tier", "d"}}) {
        expected[1].Add(label.Name(), label.Value());
    }

    std::sort(expected[1].begin(), expected[1].end());

    AssertLabelsEqual(transformer.Transform(origin), expected);

    origin.clear();
    origin.Add(TStringBuf("prj"), TStringBuf("a"));

    expected.clear();
    expected.emplace_back(&pool);
    expected[0].Add(TStringBuf("prj"), TStringBuf{NYasm::AGGREGATED_MARKER});

    AssertLabelsEqual(transformer.Transform(origin), expected);
}

TEST(Transformer, parseAndTransform) {
    TString rulsConf = NResource::Find("/rules1.conf");

    {
        TLabelPool pool;

        auto rules = GetYasmAggrRulesForProject(rulsConf, "balancer", pool);
        TYasmMetricTransformer transformer(rules);

        TLabels origin(&pool);
        origin.Add(TStringBuf("ctype"), TStringBuf("a"));
        origin.Add(TStringBuf("geo"), TStringBuf("b"));
        origin.Add(TStringBuf("prj"), TStringBuf("c"));
        origin.Add(TStringBuf("tier"), TStringBuf("d"));

        TVector<TLabels> expected;
        expected.emplace_back(&pool);
        expected.emplace_back(&pool);

        expected[0].Add(TStringBuf("ctype"), TStringBuf{NYasm::AGGREGATED_MARKER});
        expected[0].Add(TStringBuf("geo"), TStringBuf{NYasm::AGGREGATED_MARKER});
        expected[0].Add(TStringBuf("prj"), TStringBuf{NYasm::AGGREGATED_MARKER});
        expected[0].Add(TStringBuf("tier"), TStringBuf{NYasm::AGGREGATED_MARKER});

        std::sort(expected[0].begin(), expected[0].end());

        expected[1].Add(TStringBuf("ctype"), TStringBuf("a"));
        expected[1].Add(TStringBuf("geo"), TStringBuf("b"));
        expected[1].Add(TStringBuf("prj"), TStringBuf{NYasm::AGGREGATED_MARKER});
        expected[1].Add(TStringBuf("tier"), TStringBuf("d"));

        std::sort(expected[1].begin(), expected[1].end());

        AssertLabelsEqual(transformer.Transform(origin), expected);
    }

    {
        TLabelPool pool;
        auto rules = GetYasmAggrRulesForProject(rulsConf, "solomon", pool);
        TYasmMetricTransformer transformer(rules);

        TLabels origin(&pool);
        origin.Add(TStringBuf("ctype"), TStringBuf("a"));
        origin.Add(TStringBuf("geo"), TStringBuf("b"));
        origin.Add(TStringBuf("prj"), TStringBuf("c"));
        origin.Add(TStringBuf("tier"), TStringBuf("d"));

        TVector<TLabels> expected;
        expected.emplace_back(&pool);

        expected[0].Add(TStringBuf("ctype"), TStringBuf{NYasm::AGGREGATED_MARKER});
        expected[0].Add(TStringBuf("geo"), TStringBuf{NYasm::AGGREGATED_MARKER});
        expected[0].Add(TStringBuf("prj"), TStringBuf{NYasm::AGGREGATED_MARKER});
        expected[0].Add(TStringBuf("tier"), TStringBuf{NYasm::AGGREGATED_MARKER});

        std::sort(expected[0].begin(), expected[0].end());

        AssertLabelsEqual(transformer.Transform(origin), expected);
    }
}


} // namespace NSolomon::NIngestor
