#include <library/cpp/testing/gtest/gtest.h>

#include "solomon/services/ingestor/lib/shard/yasm_aggregator.h"

// TODO: remove
#include "solomon/services/ingestor/lib/shard/yasm_aggregator.cpp"

#include <library/cpp/monlib/metrics/summary_collector.h>

#include <limits>
#include <cmath>

using NMonitoring::TLabels;
using NMonitoring::EMetricType;
using NMonitoring::ISummaryDoubleSnapshotPtr;
using NMonitoring::TLogHistogramSnapshot;
using NMonitoring::TLogHistogramSnapshotPtr;

namespace NSolomon::NIngestor {

using TSummaryAggregate = TAggregate<ISummaryDoubleSnapshotPtr>;

ISummaryDoubleSnapshotPtr SummaryOf(const TVector<double>& values) {
    NMonitoring::TSummaryDoubleCollector collector;
    for (auto value: values) {
        collector.Collect(value);
    }
    return collector.Snapshot();
}

void CompareHists(IHistogramSnapshot* lhs, IHistogramSnapshot* rhs) {
    ASSERT_EQ(lhs->Count(), rhs->Count());
    for (size_t i = 0; i < lhs->Count(); ++i) {
        EXPECT_EQ(lhs->UpperBound(i), rhs->UpperBound(i));
        EXPECT_EQ(lhs->Value(i), rhs->Value(i));
    }
}

void CompareSummary(const ISummaryDoubleSnapshotPtr& summary, const TSummaryAggregate& aggregate) {
    auto rhs = aggregate.Aggregate;
    EXPECT_DOUBLE_EQ(summary->GetSum(), rhs->GetSum());
    EXPECT_DOUBLE_EQ(summary->GetMin(), rhs->GetMin());
    EXPECT_DOUBLE_EQ(summary->GetMax(), rhs->GetMax());
    EXPECT_DOUBLE_EQ(summary->GetLast(), rhs->GetLast());
    EXPECT_EQ(summary->GetCount(), rhs->GetCount());
    EXPECT_EQ(aggregate.MetricType, NMonitoring::EMetricType::DSUMMARY);
}

void CompareSum(double sum, EMetricType type, const TSummaryAggregate& aggregate) {
    ASSERT_EQ(type, aggregate.MetricType);
    auto summary = aggregate.Aggregate;
    EXPECT_EQ(summary->GetCount(), 0u);
    EXPECT_EQ(summary->GetLast(), 0.0);
    EXPECT_EQ(summary->GetMax(), 0.0);
    EXPECT_EQ(summary->GetMin(), 0.0);
    EXPECT_DOUBLE_EQ(summary->GetSum(), sum);
}

void CompareMax(double max, EMetricType type, const TSummaryAggregate& aggregate) {
    EXPECT_EQ(type, aggregate.MetricType);
    auto summary = aggregate.Aggregate;
    EXPECT_EQ(summary->GetCount(), 0u);
    EXPECT_EQ(summary->GetLast(), 0.0);
    EXPECT_EQ(summary->GetMax(), max);
    EXPECT_EQ(summary->GetMin(), 0.0);
    EXPECT_EQ(summary->GetSum(), 0.0);
}

void CompareMin(double min, EMetricType type, const TSummaryAggregate& aggregate) {
    ASSERT_EQ(type, aggregate.MetricType);
    auto summary = aggregate.Aggregate;
    EXPECT_EQ(summary->GetCount(), 0u);
    EXPECT_EQ(summary->GetLast(), 0.0);
    EXPECT_EQ(summary->GetMax(), 0.0);
    EXPECT_EQ(summary->GetMin(), min);
    EXPECT_EQ(summary->GetSum(), 0.0);
}

void CompareLast(double last, EMetricType type, const TSummaryAggregate& aggregate) {
    ASSERT_EQ(type, aggregate.MetricType);
    auto summary = aggregate.Aggregate;
    EXPECT_EQ(summary->GetCount(), 0u);
    EXPECT_EQ(summary->GetLast(), last);
    EXPECT_EQ(summary->GetMax(), 0.0);
    EXPECT_EQ(summary->GetMin(), 0.0);
    EXPECT_EQ(summary->GetSum(), 0.0);
}

void CompareAvg(double sum, size_t count, EMetricType type, const TSummaryAggregate& aggregate) {
    ASSERT_EQ(type, aggregate.MetricType);
    auto summary = aggregate.Aggregate;
    EXPECT_EQ(summary->GetCount(), count);
    EXPECT_EQ(summary->GetLast(), 0.0);
    EXPECT_EQ(summary->GetMax(), 0.0);
    EXPECT_EQ(summary->GetMin(), 0.0);
    EXPECT_DOUBLE_EQ(summary->GetSum(), sum);
}

void CheckInvalid(TStringBuf suffix) {
    TStringStream stream;
    stream << "aaaa_name_" << suffix;

    TLabelPool pool;
    TLabels labels(&pool);
    labels.Add(TStringBuf("signal"), TStringBuf(stream.Data()));
    EMetricType type = EMetricType::GAUGE;

    TYasmAggrState state({});

    ASSERT_THROW(state.Collect(0.0, labels, type), TInvalidSuffixError);
}

struct THistogramAggregates {
    TVector<TAggregate<NMonitoring::IHistogramSnapshotPtr>> HistAggregates;
    TVector<TAggregate<NMonitoring::TLogHistogramSnapshotPtr>> LogHistAggregates;
};

class TCollectingAggregatesConsumer: public IYasmAggregatesProcessorConsumer {
public:
    TVector<TAggregate<NMonitoring::ISummaryDoubleSnapshotPtr>> ReleaseSummary() {
        return std::move(Summary_);
    }

    THistogramAggregates ReleaseHists() {
        return std::move(Hists_);
    }

private:
    TVector<TAggregate<NMonitoring::ISummaryDoubleSnapshotPtr>> Summary_;
    THistogramAggregates Hists_;

    void OnSummary(TAggregate<NMonitoring::ISummaryDoubleSnapshotPtr> aggr) override {
        Summary_.emplace_back(std::move(aggr));
    }

    void OnHistogram(TAggregate<NMonitoring::IHistogramSnapshotPtr> aggr) override {
        Hists_.HistAggregates.emplace_back(std::move(aggr));
    }

    void OnLogHistogram(TAggregate<NMonitoring::TLogHistogramSnapshotPtr> aggr) override {
        Hists_.LogHistAggregates.emplace_back(std::move(aggr));
    }
};

TVector<TAggregate<NMonitoring::ISummaryDoubleSnapshotPtr>> ReleaseSummaryAggregates(TYasmAggrState& state) {
    auto aggrs = state.ReleaseAggregates();
    TCollectingAggregatesConsumer consumer;
    aggrs->Consume(consumer);
    return consumer.ReleaseSummary();
}

THistogramAggregates ReleaseHistTypeAggregates(TYasmAggrState& state) {
    auto aggrs = state.ReleaseAggregates();
    TCollectingAggregatesConsumer consumer;
    aggrs->Consume(consumer);
    return consumer.ReleaseHists();
}


TEST(TYasmAggrState, Simple) {
    TYasmAggrState state({});
    auto aggrs = ReleaseSummaryAggregates(state);
    ASSERT_TRUE(aggrs.empty());
}

TEST(TYasmAggrState, Summ) {
    TYasmAggrState state({});
    TLabelPool pool;

    TLabels labels(&pool);
    labels.Add(TStringBuf("signal"), TStringBuf("aaa_name_summ"));
    EMetricType type = EMetricType::GAUGE;

    double sum = 0;
    for (size_t i = 0; i < 100; ++i) {
        state.Collect(static_cast<double>(i), labels, type);
        sum += static_cast<double>(i);
    }

    auto aggrs = ReleaseSummaryAggregates(state);
    ASSERT_EQ(aggrs.size(), 1u);

    CompareSum(sum, type, aggrs[0]);

    labels.clear();
    labels.Add(TStringBuf("signal"), TStringBuf("aaa_name_ammm"));
    for (size_t i = 0; i < 100; ++i) {
        state.Collect(static_cast<double>(i), labels, type);
    }

    aggrs = ReleaseSummaryAggregates(state);
    ASSERT_EQ(aggrs.size(), 1u);
    CompareSum(sum, type, aggrs[0]);
}

TEST(TYasmAggrState, Max) {
    TYasmAggrState state({});

    TLabelPool pool;
    TLabels labels(&pool);
        labels.Add(TStringBuf("signal"), TStringBuf("aaa_name_max"));
    EMetricType type = EMetricType::GAUGE;

    for (size_t i = 0; i < 100; ++i) {
        state.Collect(static_cast<double>(i % 10), labels, type);
    }

    auto aggrs = ReleaseSummaryAggregates(state);
    ASSERT_EQ(aggrs.size(), 1u);

    CompareMax(static_cast<double>(9), type, aggrs[0]);

    labels.clear();
        labels.Add(TStringBuf("signal"), TStringBuf("aaa_name_axxx"));
    for (size_t i = 0; i < 100; ++i) {
        state.Collect(static_cast<double>(i % 10), labels, type);
    }

    aggrs = ReleaseSummaryAggregates(state);
    ASSERT_EQ(aggrs.size(), 1u);
    CompareMax(static_cast<double>(9), type, aggrs[0]);
}

TEST(TYasmAggrState, Min) {
    TYasmAggrState state({});

    TLabelPool pool;
    TLabels labels(&pool);
        labels.Add(TStringBuf("signal"), TStringBuf("aaa_name_annn"));

    EMetricType type = EMetricType::GAUGE;

    for (size_t i = 1; i < 100; ++i) {
        state.Collect(static_cast<double>(i % 10), labels, type);
    }

    auto aggrs = ReleaseSummaryAggregates(state);
    ASSERT_EQ(aggrs.size(), 1u);

    CompareMin(static_cast<double>(0), type, aggrs[0]);
}

TEST(TYasmAggrState, Last) {
    TYasmAggrState state({});
    TLabelPool pool;
    TLabels labels(&pool);
        labels.Add(TStringBuf("signal"), TStringBuf("aaa_name_attt"));
    EMetricType type = EMetricType::GAUGE;

    for (size_t i = 1; i < 100; ++i) {
        state.Collect(static_cast<double>(i), labels, type);
    }

    auto aggrs = ReleaseSummaryAggregates(state);
    ASSERT_EQ(aggrs.size(), 1u);

    CompareLast(static_cast<double>(99), type, aggrs[0]);
}

TEST(TYasmAggrState, Avg) {
    TYasmAggrState state({});
    TLabelPool pool;
    TLabels labels(&pool);
        labels.Add(TStringBuf("signal"), TStringBuf("aaa_name_avvv"));
    EMetricType type = EMetricType::GAUGE;

    double sum = 0;
    for (size_t i = 0; i < 100; ++i) {
        state.Collect(static_cast<double>(i), labels, type);
        sum += static_cast<double>(i);
    }

    auto aggrs = ReleaseSummaryAggregates(state);;
    ASSERT_EQ(aggrs.size(), 1u);

    CompareAvg(sum, 100u, type, aggrs[0]);
}

TEST(TYasmAggrState, LogHist) {
    TLogHistogramSnapshotPtr logHist = MakeIntrusive<TLogHistogramSnapshot>(1.5, 1u, -1, TVector<double>{0.0, 1.0, 1.0});
    TLabelPool pool;
    TLabels labels(&pool);
        labels.Add(TStringBuf("signal"), TStringBuf("aaa_name_ahhh"));
    TYasmAggrState state({});
    for (size_t i = 0; i < 100; ++i) {
        state.Collect(logHist.Get(), labels, EMetricType::LOGHIST);
    }

    auto aggrs = ReleaseHistTypeAggregates(state).LogHistAggregates;
    ASSERT_EQ(aggrs.size(), 1u);
    auto aggregate = aggrs[0];
    auto histSnapshot = aggregate.Aggregate;
    ASSERT_EQ(aggregate.MetricType, EMetricType::LOGHIST);
    EXPECT_EQ(histSnapshot->ZerosCount(), 100u);
    EXPECT_EQ(histSnapshot->StartPower(), -1);
    EXPECT_EQ(histSnapshot->Count(), 3u);
    EXPECT_DOUBLE_EQ(histSnapshot->Bucket(0), 0.0);
    EXPECT_DOUBLE_EQ(histSnapshot->Bucket(1), 100.0);
    EXPECT_DOUBLE_EQ(histSnapshot->Bucket(2), 100.0);
}

TEST(TYasmAggrState, SummarySum) {
    TLabelPool pool;
    TLabels labels(&pool);

    labels.Add(TStringBuf("signal"), TStringBuf("aaa_name_summ"));
    TYasmAggrState state({});
    state.Collect(SummaryOf({1, -1, 3, 5}).Get(), labels, EMetricType::DSUMMARY);
    state.Collect(SummaryOf({2, -5, 4, 0}).Get(), labels, EMetricType::DSUMMARY);
    state.Collect(SummaryOf({3, 3, 5}).Get(), labels, EMetricType::DSUMMARY);

    auto aggrs = ReleaseSummaryAggregates(state);
    ASSERT_EQ(aggrs.size(), 1u);

    CompareSummary(SummaryOf({1, -1, 3, 5, 2, -5, 4, 0, 3, 3, 5}), aggrs[0]);
}

TEST(TYasmAggrState, AvgRollup) {
    TLabelPool pool;
    TLabels labels(&pool);
        labels.Add(TStringBuf("signal"), TStringBuf("aaa_name_tmmv"));

    TYasmAggrState state({});
    EMetricType type = EMetricType::GAUGE;
    double sum = 0;
    for (size_t i = 0; i < 100; ++i) {
        state.Collect(static_cast<double>(i), labels, type);
        sum += static_cast<double>(i);
    }

    auto aggrs = ReleaseSummaryAggregates(state);
    ASSERT_EQ(aggrs.size(), 1u);

    CompareAvg(sum, 1u, type, aggrs[0]);
}

TEST(TYasmAggrState, SummaryLast) {
    TLabelPool pool;
    TLabels labels(&pool);
        labels.Add(TStringBuf("signal"), TStringBuf("aaa_name_attt"));
    TYasmAggrState state({});
    state.Collect(SummaryOf({1, -1, 3, 5}).Get(), labels, EMetricType::DSUMMARY);
    state.Collect(SummaryOf({2, -5, 4, 0}).Get(), labels, EMetricType::DSUMMARY);
    state.Collect(SummaryOf({3, 3, 5}).Get(), labels, EMetricType::DSUMMARY);

    auto aggrs = ReleaseSummaryAggregates(state);
    ASSERT_EQ(aggrs.size(), 1u);

    CompareSummary(SummaryOf({3, 3, 5}), aggrs[0]);
}

TEST(TYasmAggrState, HistogramBounds) {
    THistogramCollectors::THistogramMergeCollector Merger;

    TBucketBounds b1{1.0, 4.0, 6.0, 9.0, 15.0, 20.0};
    TBucketValues v1{0, 0, 0, 0, 0, 0};
    auto h1 = NMonitoring::ExplicitHistogramSnapshot(b1, v1);


    TBucketBounds b2{2.0, 4.0, 5.0, 9.0, 10.0, 15.0, 21.0};
    TBucketValues v2{0, 0, 0, 0, 0, 0, 0};
    auto h2 = NMonitoring::ExplicitHistogramSnapshot(b2, v2);

    Merger.Collect(h1.Get());
    Merger.Collect(h2.Get());

    auto snapshot = Merger.Snapshot();

    TBucketBounds b3{1.0, 2.0, 4.0, 5.0, 6.0, 9.0, 10.0, 15.0, 20.0, 21.0};
    TBucketValues v3{0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    auto h3 = NMonitoring::ExplicitHistogramSnapshot(b3, v3);

    CompareHists(snapshot.Get(), h3.Get());
}

TEST(TYasmAggrState, SimpleMerge) {
    THistogramCollectors::THistogramMergeCollector Merger;

    TBucketBounds b1{1.0, 4.0, 6.0, 9.0, 15.0, 20.0};
    TBucketValues v1{1, 1, 1, 1, 1, 1};
    auto h1 = NMonitoring::ExplicitHistogramSnapshot(b1, v1);

    Merger.Collect(h1.Get());
    Merger.Collect(h1.Get());

    auto h2 = NMonitoring::ExplicitHistogramSnapshot(b1, {2, 2, 2, 2, 2, 2});

    CompareHists(Merger.Snapshot().Get(), h2.Get());
}

TEST(TYasmAggrState, HistMerge1) {
    THistogramCollectors::THistogramMergeCollector Merger;
    TBucketBounds b1{10., 20., 50., 60., NMonitoring::HISTOGRAM_INF_BOUND};
    TBucketValues v1{0, 2, 6, 1, 0};

    TBucketBounds b2{5., 10., 15., 30., 60., NMonitoring::HISTOGRAM_INF_BOUND};
    TBucketValues v2{0, 1, 3, 3, 3, 0};

    auto h1 = NMonitoring::ExplicitHistogramSnapshot(b1, v1);
    auto h2 = NMonitoring::ExplicitHistogramSnapshot(b2, v2);

    Merger.Collect(h1.Get());
    Merger.Collect(h2.Get());

    auto h = NMonitoring::ExplicitHistogramSnapshot(
            {5., 10., 15., 20., 30., 50., 60., NMonitoring::HISTOGRAM_INF_BOUND},
            {0, 1, 4, 2, 4, 6, 2, 0});

    CompareHists(Merger.Snapshot().Get(), h.Get());
}

TEST(TYasmAggrState, HistMerge2) {
    THistogramCollectors::THistogramMergeCollector Merger;
    TBucketBound after50 = std::nextafter(50., std::numeric_limits<double>::max());

    auto h1 = NMonitoring::ExplicitHistogramSnapshot(
            {10., 20., 30., 50., after50, NMonitoring::HISTOGRAM_INF_BOUND},
            {0, 2, 6, 12, 1, 0});
    auto h2 = NMonitoring::ExplicitHistogramSnapshot(
            {10., 30., 40., 45., 50., NMonitoring::HISTOGRAM_INF_BOUND},
            {0, 8, 3, 5, 10, 0});

    Merger.Collect(h1.Get());
    Merger.Collect(h2.Get());

    auto h = NMonitoring::ExplicitHistogramSnapshot(
            {10., 20., 30., 40., 45., 50., after50, NMonitoring::HISTOGRAM_INF_BOUND},
            {0, 6, 10, 9, 8, 13, 1, 0});

    CompareHists(Merger.Snapshot().Get(), h.Get());
}

TEST(TYasmAggrState, HistMerge3) {
    THistogramCollectors::THistogramMergeCollector Merger;
    TBucketBound after50 = std::nextafter(50., std::numeric_limits<double>::max());

    auto h1 = NMonitoring::ExplicitHistogramSnapshot(
            {10., 20., 50., after50, NMonitoring::HISTOGRAM_INF_BOUND},
            {0, 2, 6, 1, 0});
    auto h2 = NMonitoring::ExplicitHistogramSnapshot(
            {5., 10., 15., 30., 60., NMonitoring::HISTOGRAM_INF_BOUND},
            {0, 1, 3, 3, 3, 0});

    Merger.Collect(h1.Get());
    Merger.Collect(h2.Get());

    auto h = NMonitoring::ExplicitHistogramSnapshot(
            {5., 10., 15., 20., 30., 50., after50, 60., NMonitoring::HISTOGRAM_INF_BOUND},
            {0, 1, 4, 2, 4, 6, 1, 1, 0});

    CompareHists(Merger.Snapshot().Get(), h.Get());
}

TEST(TYasmAggrState, Compress) {
    NMonitoring::TBucketBounds bounds;
    NMonitoring::TBucketValues values;

    for (size_t i = 0; i < 1000u; ++i) {
        bounds.emplace_back(i);
        values.emplace_back(!(i % 100));
    }

    bounds.emplace_back(NMonitoring::HISTOGRAM_INF_BOUND);
    values.emplace_back(0);

    THistogramCollectors::TUgramCompressor::Compress(bounds, values);

    auto actb = NMonitoring::TBucketBounds{0., 100., 200., 300., 400., 500., 600., 700., 800., 900., NMonitoring::HISTOGRAM_INF_BOUND};
    auto actv = NMonitoring::TBucketValues{1u, 1u, 1u, 1u, 1u, 1u, 1u, 1u, 1u, 1u, 0u};
    EXPECT_EQ(bounds, actb);
    EXPECT_EQ(values, actv);
}

double NextAfter(double v) {
    return std::nextafter(v, std::numeric_limits<double>::max());
}

NZoom::NHgram::TUgramBucket Ub(const double lowerBound, const double upperBound, const double weight) {
    return NZoom::NHgram::TUgramBucket(lowerBound, upperBound, weight);
}

TEST(TYasmAggrState, Convert) {
    TBucketBounds bounds{0, 5, 10, NextAfter(10.), 15, NMonitoring::HISTOGRAM_INF_BOUND};
    TBucketValues values{1, 2, 3, 4, 5, 6};

    auto ugram = THistogramCollectors::ToUgramBuckets(bounds, values);

    NZoom::NHgram::TUgramBuckets expectedUgram{
            Ub(std::numeric_limits<double>::lowest(), 0, 1),
            Ub(0, 5, 2),
            Ub(5, 10, 3),
            Ub(10, 10, 4),
            Ub(10, 15, 5),
            Ub(15, NMonitoring::HISTOGRAM_INF_BOUND, 6),
    };

    EXPECT_EQ(ugram, expectedUgram);

    auto [convertBounds, convertValues] = THistogramCollectors::Ugram2HistBounds(ugram);

    EXPECT_EQ(bounds, convertBounds);
    EXPECT_EQ(values, convertValues);
}

TEST(TYasmAggrState, Invalid) {
    CheckInvalid("assx");
    CheckInvalid("sum");
    CheckInvalid("axxxx");
    CheckInvalid("xxx");
}


} // namespace NSolomon::NIngestor
