#pragma once

#include "yasm_metric_transformer.h"
#include "processing_error_listner.h"

#include <library/cpp/monlib/metrics/labels.h>
#include <library/cpp/monlib/metrics/log_histogram_snapshot.h>
#include <library/cpp/monlib/metrics/summary_snapshot.h>
#include <library/cpp/monlib/metrics/histogram_snapshot.h>
#include <library/cpp/monlib/metrics/metric_type.h>

#include <util/generic/vector.h>

namespace NSolomon::NIngestor {

struct TInvalidSuffixError: public yexception {};
struct TInvalidValueTypeError: public yexception {};

inline EInvalidMetricReason ClassifyYasmAggrError() {
    try {
        throw;
    } catch (const TInvalidSuffixError&) {
        return EInvalidMetricReason::INVALID_AGGR_SUFFIX;
    } catch (const TInvalidValueTypeError&) {
        return EInvalidMetricReason::INVALID_VALUE_TYPE;
    } catch (...) {
        return EInvalidMetricReason::UNKNOWN_ERROR;
    }
}

template <class TSnapshot>
struct TAggregate {
    TLabels Labels;
    TSnapshot Aggregate;
    NMonitoring::EMetricType MetricType;
    size_t Count;

    TAggregate(TLabels labels, TSnapshot snapshot, NMonitoring::EMetricType type, size_t count)
            : Labels(std::move(labels))
            , Aggregate(std::move(snapshot))
            , MetricType(type)
            , Count(count)
    {}
};

class IYasmAggregatesProcessorConsumer {
public:
    virtual void OnSummary(TAggregate<NMonitoring::ISummaryDoubleSnapshotPtr>) = 0;

    virtual void OnHistogram(TAggregate<NMonitoring::IHistogramSnapshotPtr>) = 0;

    virtual void OnLogHistogram(TAggregate<NMonitoring::TLogHistogramSnapshotPtr>) = 0;

    virtual ~IYasmAggregatesProcessorConsumer() = default;
};

class IYasmAggregates {
public:
    virtual void Consume(IYasmAggregatesProcessorConsumer& consumer) = 0;

    virtual ~IYasmAggregates() = default;
};

using IYasmAggregatesPtr = THolder<IYasmAggregates>;

class TYasmAggrState {
public:
    TYasmAggrState(TVector<TYasmAggrRule> rules);

    TYasmMetricTransformer* GetTransformer();

    void Collect(double value, const TLabels& labels, NMonitoring::EMetricType metricType);

    void Collect(
            NMonitoring::TLogHistogramSnapshot* value,
            const TLabels& labels,
            NMonitoring::EMetricType metricType);

    void Collect(
            NMonitoring::ISummaryDoubleSnapshot* value,
            const TLabels& labels,
            NMonitoring::EMetricType metricType);

    void Collect(
            NMonitoring::IHistogramSnapshot* value,
            const TLabels& labels,
            NMonitoring::EMetricType metricType);

    void CollectTransform(double value, const TLabels& labels, NMonitoring::EMetricType metricType);

    void CollectTransform(
            NMonitoring::TLogHistogramSnapshot* value,
            const TLabels& labels,
            NMonitoring::EMetricType metricType);

    void CollectTransform(
            NMonitoring::ISummaryDoubleSnapshot* value,
            const TLabels& labels,
            NMonitoring::EMetricType metricType);

    void CollectTransform(
            NMonitoring::IHistogramSnapshot* value,
            const TLabels& labels,
            NMonitoring::EMetricType metricType);

    size_t SizeBytes() const;

    IYasmAggregatesPtr ReleaseAggregates();

    ~TYasmAggrState();

private:
    class TImpl;
    THolder<TImpl> Impl_;
};

NMonitoring::ISummaryDoubleSnapshotPtr GaugeToSummary(const TLabels& labels, double gauge);

} // namespace NSolomon::NIngestor
