#include "shard_actor.h"
#include "tasks_counter.h"
#include "metrics.h"

#include <solomon/services/ingestor/lib/shard/shard.h>
#include <solomon/services/ingestor/lib/shard_config/shard_config.h>
#include <solomon/services/ingestor/lib/thread_pool/pool.h>

#include <solomon/services/dataproxy/lib/memstore/watcher.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/grpc/status/code.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/steady_timer/steady_timer.h>
#include <solomon/libs/cpp/yasm/constants/interval.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/actorid.h>
#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/actors/core/event.h>
#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/log.h>
#include <library/cpp/grpc/client/grpc_client_low.h>
#include <library/cpp/monlib/metrics/labels.h>
#include <library/cpp/threading/future/future.h>

#include <util/datetime/base.h>
#include <util/generic/ptr.h>
#include <util/generic/scope.h>
#include <util/generic/yexception.h>

#include <type_traits>
#include <queue>

namespace NSolomon::NIngestor {
namespace {

using namespace NActors;
using NSolomon::NDataProxy::TMemStoreWatcherEvents;
using yandex::solomon::common::UrlStatusType;

struct TDocumentData: private TNonCopyable, public TIntrusiveListItem<TDocumentData> {
    TDocumentData(TActorId handlerId, ui64 cookie) noexcept
        : HandlerId(handlerId)
        , Cookie{cookie}
    {
    }

    size_t SizeBytes() const {
        size_t res = sizeof(*this);
        for (const auto& shardData: ShardData) {
            if (shardData) {
                res += sizeof(*shardData);
                res += shardData->Host.capacity();
                res += shardData->Data.capacity();
                for (const auto& l: shardData->HostOptLabels) {
                    auto* label = static_cast<const NMonitoring::TLabel*>(&l);
                    res += label->NameStr().capacity();
                    res += label->ValueStr().capacity();
                    res += sizeof(*label);
                }
                res += (shardData->HostOptLabels.capacity() - shardData->HostOptLabels.size()) * sizeof(NMonitoring::TLabel);
            }
        }
        res += ShardData.capacity() * sizeof(std::unique_ptr<TShardData>);
        return res;
    }

    std::vector<std::unique_ptr<TShardData>> ShardData;
    TActorId HandlerId;
    ui64 Cookie;
};

struct TShardActorPrivateEvents: private TPrivateEvents {
    enum {
        EvOnProcessedDataCallback = SpaceBegin,
        EvOnAllDataProcessed,
        EvOnEndOfInterval,
        EvResolveShard,
        End,
    };

    static_assert(End < SpaceEnd, "too many event types");

    struct TEvOnProcessedDataCallback: TEventLocal<TEvOnProcessedDataCallback, EvOnProcessedDataCallback> {
        TEvOnProcessedDataCallback(TDataProcessor::TProcessingResult processingResult, TActorId handlerId)
            : ProcessingResult(std::move(processingResult))
            , HandlerId(handlerId)
        {
        }

        TDataProcessor::TProcessingResult ProcessingResult;
        TActorId HandlerId;
    };

    struct TEvOnAllDataProcessed: TEventLocal<TEvOnAllDataProcessed, EvOnAllDataProcessed> {
    };

    struct TEvOnEndOfInterval: TEventLocal<TEvOnEndOfInterval, EvOnEndOfInterval> {
    };

    struct TEvResolveShard: public NActors::TEventLocal<TEvResolveShard, EvResolveShard> {
    };
};

class TShardActor: public TActorBootstrapped<TShardActor> {
public:
    TShardActor(
            TShardConfig shardConfig,
            IThreadPool* processorsExecutor,
            std::shared_ptr<IClusterRpc<NMemStore::IMemStoreRpc>> memStoreRpc,
            TActorId memStoreClusterWatcher,
            const std::shared_ptr<TShardMeteringMetricsRepository>& meteringMetricsRepo,
            const TString& yasmPrefix,
            NMonitoring::IMetricRegistry* registry)
        : ShardNumId_(shardConfig.ShardNumId)
        , ShardStrId_(shardConfig.ShardStrId)
        , MemStoreRpc_{std::move(memStoreRpc)}
        , MemStoreClusterWatcher_{memStoreClusterWatcher}
        , ShardMeteringContext_{
                    meteringMetricsRepo
                    ? meteringMetricsRepo->GetContext(shardConfig.Key.ProjectId, shardConfig.ShardStrId)
                    : nullptr
        }
        , IsYasmShard_(IsYasmShard(shardConfig))
        , Interval_(GetInterval(shardConfig.Interval, IsYasmShard_))
        , Metrics_(shardConfig.ShardStrId, registry)
        , ShardKey_{shardConfig.Key}
        , Shard_{
            std::move(shardConfig),
            LabelPool_,
            CreateThreadPoolProxyWithMetering(
                    processorsExecutor,
                    ShardMeteringContext_),
            [this]() {
                TasksCounter_->OnFinish();
            },
            yasmPrefix
        }
        , Timer_{}
    {
        if (ShardMeteringContext_) {
            ShardMeteringContext_->ShardNumId = ShardNumId_;
        }
    }

    void Bootstrap() {
        Become(&TThis::StateWork);
        InitTasksCounter();
        StartInterval(TInstant::Now());

        ResolveShardLocation();
    }

    STATEFN(StateWork) {
        Timer_.Reset();

        Y_DEFER {
            if (ShardMeteringContext_) {
                ShardMeteringContext_->AddCpuTime(Timer_.Step());
            }
        };

        switch (ev->GetTypeRewrite()) {
            hFunc(TEvents::TEvPoison, OnPoison);
            hFunc(TShardActorEvents::TUpdateConfig, OnUpdateConfig);
            hFunc(TShardActorPrivateEvents::TEvOnAllDataProcessed, OnAllDataProcessed);
            hFunc(TShardActorEvents::TProcessData, OnProcessData);
            hFunc(TShardActorPrivateEvents::TEvOnProcessedDataCallback, OnProcessedDataCallback);
            sFunc(TShardActorEvents::TCalcSizeBytes, OnCalcSizeBytes)
            cFunc(TShardActorPrivateEvents::EvOnEndOfInterval, OnEndOfInterval);
            hFunc(TMemStoreWatcherEvents::TResolveResult, OnMemStoreShardResolved);
            sFunc(TShardActorPrivateEvents::TEvResolveShard, ResolveShardLocation);
        }
    }

private:
    void ResolveShardLocation() {
        Send(MemStoreClusterWatcher_, new TMemStoreWatcherEvents::TResolve{ {ShardNumId_} });
    }

    void OnMemStoreShardResolved(const TMemStoreWatcherEvents::TResolveResult::TPtr& evPtr) {
        if (auto& loc = evPtr->Get()->Locations; !loc.empty()) {
            MemStoreNode_ = MemStoreRpc_->Get(loc[0].Address);
        } else {
            Schedule(TDuration::Seconds(5), new TShardActorPrivateEvents::TEvResolveShard);
        }
    }

    static TDuration GetInterval(const TDuration& interval, bool isYasmShard) {
        if (interval.Seconds() == 0) {
            return isYasmShard ? NYasm::YASM_INTERVAL : DEFAULT_INTERVAL;
        } else {
            return interval;
        }
    }

    auto SubscribeToProcessedData(
            NThreading::TFuture<TDataProcessor::TProcessingResult>& future,
            TActorId handlerId, ui64 cookie, size_t batchSize)
    {
        auto numId = ShardNumId_;
        auto selfId = SelfId();
        auto* actorSystemPtr = NActors::TActorContext::ActorSystem();

        return future.Subscribe([=] (auto f) {
            try {
                auto* event = new TShardActorPrivateEvents::TEvOnProcessedDataCallback{f.ExtractValue(), handlerId};
                actorSystemPtr->Send(new IEventHandle{selfId, {}, event, 0, cookie});
            } catch (...) {
                TString errorMsg = "Parsing or validation error: " + CurrentExceptionMessage();
                MON_ERROR_C(*actorSystemPtr, ShardActor, errorMsg);

                auto event = std::make_unique<TShardActorEvents::TProcessingResult>(numId);
                event->Statuses.resize(batchSize);
                for (auto& s: event->Statuses) {
                    s = {UrlStatusType::PARSE_ERROR, errorMsg, 0};
                }

                actorSystemPtr->Send(new IEventHandle(handlerId, selfId, event.release(), 0, cookie));
            }
        });
    }

    TInstant AlignTimeByGrid(TInstant rawTime) {
        auto timeInSeconds = rawTime.Seconds();
        return TInstant::Seconds(timeInSeconds - timeInSeconds % Interval_.Seconds());
    }

    void OnCalcSizeBytes() {
        if (ShardMeteringContext_) {
            size_t currentSize = SizeBytes();
            MON_INFO(ShardActor, "current size of " << ShardStrId_ << " is " << currentSize);
            ShardMeteringContext_->SetMemoryBytes(currentSize);
        }
    }

    void StartInterval(TInstant startTime) {
        while (!NextWindowQueue_.Empty()) {
            std::unique_ptr<TDocumentData> documentData{NextWindowQueue_.PopFront()};

            TasksCounter_->OnStart();
            size_t batchSize = documentData->ShardData.size();
            auto f = Shard_.ProcessData(
                    std::move(documentData->ShardData),
                    EndOfCurrentWindow_,
                    EndOfSkipWindow_ - EndOfNextWindow_);
            SubscribeToProcessedData(f, documentData->HandlerId, documentData->Cookie, batchSize);
        }

        SkipWindowQueue_.Swap(NextWindowQueue_);

        startTime = AlignTimeByGrid(startTime);
        CurrentWindowStart_ = startTime;
        EndOfCurrentWindow_ = TInstant::Seconds(Interval_.Seconds() + startTime.Seconds());
        EndOfNextWindow_ = TInstant::Seconds(Interval_.Seconds() + EndOfCurrentWindow_.Seconds());
        EndOfSkipWindow_ = TInstant::Seconds(Interval_.Seconds() + EndOfNextWindow_.Seconds());
        IsWindowClosed_ = false;

        Schedule(EndOfCurrentWindow_ - TInstant::Now(), new TShardActorPrivateEvents::TEvOnEndOfInterval);
    }

    void OnEndOfInterval() {
        MON_TRACE(ShardActor, "begin new interval\n");
        IsWindowClosed_ = true;
        TasksCounter_->SetWait();
    }

    void OnProcessData(TShardActorEvents::TProcessData::TPtr& ev) {
        auto* req = ev->Get();
        ui64 cookie = ev->Cookie;

        if (ShardMeteringContext_) {
            size_t size = 0;
            for (auto& d: req->Data) {
                size += d->Data.size();
            }
            ShardMeteringContext_->AddNetworkRxBytes(size);
        }

        // split batch into 3 different lists where data is fall into:
        //   1) current window
        //   2) next window
        //   3) skip window (next next window)
        std::unique_ptr<TDocumentData> skipDoc;
        std::unique_ptr<TDocumentData> nextDoc;

        for (size_t i = 0; i < req->Data.size(); i++) {
            TInstant time = req->Data[i]->TimesMillis;

            Metrics_.IncInputBytes(req->Data[i]->Data.size());

            if (IsWindowClosed_) {
                if (EndOfNextWindow_ <= time && time < EndOfSkipWindow_) {
                    if (!skipDoc) {
                        skipDoc = std::make_unique<TDocumentData>(req->HandlerId, 0);
                    }
                    skipDoc->ShardData.push_back(std::move(req->Data[i]));
                } else {
                    if (time >= EndOfNextWindow_) {
                        ReportLargeTs(time);
                    }
                    if (!nextDoc) {
                        nextDoc = std::make_unique<TDocumentData>(req->HandlerId, 0);
                    }
                    nextDoc->ShardData.push_back(std::move(req->Data[i]));
                }
            } else if (EndOfCurrentWindow_ <= time && time < EndOfNextWindow_) {
                if (!nextDoc) {
                    nextDoc = std::make_unique<TDocumentData>(req->HandlerId, 0);
                }
                nextDoc->ShardData.push_back(std::move(req->Data[i]));
            } else {
                if (time >= EndOfCurrentWindow_) {
                    ReportLargeTs(time);
                }
            }
        }

        auto hasDataInCurrentWindow = AnyOf(req->Data, [](const auto& d) {
            return static_cast<bool>(d);
        });

        bool hasDataInNextWindow = (bool) nextDoc;

        if (nextDoc) {
            if (!hasDataInCurrentWindow) {
                nextDoc->Cookie = cookie;
            }
            NextWindowQueue_.PushBack(nextDoc.release());
            MON_TRACE(ShardActor, "Push data into the next queue");
        }

        if (skipDoc) {
            if (!hasDataInCurrentWindow && !hasDataInNextWindow) {
                skipDoc->Cookie = cookie;
            }
            SkipWindowQueue_.PushBack(skipDoc.release());
            MON_TRACE(ShardActor, "Push data into the skip queue");
        }

        if (hasDataInCurrentWindow) {
            TasksCounter_->OnStart();
            auto f = Shard_.ProcessData(
                std::move(req->Data),
                CurrentWindowStart_,
                EndOfCurrentWindow_ - CurrentWindowStart_);
            SubscribeToProcessedData(f, req->HandlerId, cookie, req->Data.size());

//        MON_TRACE(
//            ShardActor,
//            "RequestDataBegin:\n" << req->Request.DebugString() << "RequestDataEnd" << Endl);
        }
    }

    void OnProcessedDataCallback(TShardActorPrivateEvents::TEvOnProcessedDataCallback::TPtr& ev) {
        auto handlerId = ev->Get()->HandlerId;
        auto& processingResult = ev->Get()->ProcessingResult;
        auto cookie = ev->Cookie;
        MON_TRACE(ShardActor, "processing done, "
                << " data size: " << processingResult.Data.size()
                << ", meta size: " << processingResult.Meta.size());

        size_t errors = 0;
        size_t metricsWritten = 0;

        for (const auto& s: processingResult.Statuses) {
            if (!s.Message.empty()) {
                errors++;
            }
            metricsWritten += s.MetricsWritten;
        }

        Metrics_.IncMetrics(metricsWritten);
        Metrics_.IncOutputBytes(processingResult.Data.size() + processingResult.Meta.size());

        if (errors > 0) {
            TStringBuilder errorMessage;
            errorMessage << "processing errors (" << errors << "):\n";
            for (auto& s: processingResult.Statuses) {
                if (!s.Message.empty()) {
                    errorMessage << s.Message << '\n';
                }
            }
            MON_WARN(ShardActor, errorMessage);
        }

        if (metricsWritten > 0 && MemStoreNode_ != nullptr) {
            RunMemStoreQuery(std::move(processingResult), handlerId, cookie);
        } else {
            if (!MemStoreNode_) {
                MON_WARN(ShardActor, "MemStoreClient not found");
            }

            auto* event = new TShardActorEvents::TProcessingResult(ShardNumId_);
            event->Statuses = std::move(processingResult.Statuses);
            Send(handlerId, event, 0, cookie);
        }
    }

    void RunMemStoreQuery(
        TDataProcessor::TProcessingResult processingResult,
        TActorId handlerId,
        ui64 cookie)
    {
        const TString& meta = processingResult.Meta;
        const TString& data = processingResult.Data;

        MON_DEBUG(
            ShardActor,
            "Send data to memstore. MetaData size: " << meta.size() << ". DataSize: " << data.size());

        auto totalBytes = meta.size() + data.size();
        yandex::monitoring::memstore::WriteRequest request;
        request.set_num_id(ShardNumId_);
        {
            auto* shardKey = request.mutable_shard_key();
            shardKey->set_project(ShardKey_.ProjectId);
            shardKey->set_cluster(ShardKey_.ClusterName);
            shardKey->set_service(ShardKey_.ServiceName);
        }
        request.set_data(data);
        request.set_metadata(meta);

        auto future = MemStoreNode_->Write(request);
        future.Subscribe([
            numId = ShardNumId_,
            selfId = SelfId(),
            actorSystemPtr = NActors::TActorContext::ActorSystem(),
            pr = std::move(processingResult),
            hId = handlerId,
            cookie,
            meteringCtx{ShardMeteringContext_},
            bytesTransferred = totalBytes] (auto f) mutable
        {
            TErrorOr<yandex::monitoring::memstore::WriteResponse, NGrpc::TGrpcStatus> memStoreResponse;
            try {
                memStoreResponse = f.ExtractValue();
                meteringCtx->AddNetworkTxBytes(bytesTransferred);
            } catch (...) {
                memStoreResponse = memStoreResponse.FromError(NGrpc::TGrpcStatus::Internal(CurrentExceptionMessage()));
            }

            auto event = std::make_unique<TShardActorEvents::TProcessingResult>(numId);
            if (memStoreResponse.Fail()) {
                const auto &error = memStoreResponse.Error();
                TString errorMsg = TStringBuilder{} << NGrpc::StatusCodeToString(error.GRpcStatusCode) << ": " << error.Msg;
                MON_ERROR_C(*actorSystemPtr, ShardActor, "MemStore write failed. Drop data. Error: " << errorMsg);

                auto respStatusCode = static_cast<grpc::StatusCode>(error.GRpcStatusCode);
                bool needToResolveShardLocation = !error.InternalError &&
                        (respStatusCode == grpc::StatusCode::NOT_FOUND || respStatusCode == grpc::StatusCode::UNAVAILABLE);

                if (needToResolveShardLocation) {
                    actorSystemPtr->Send(selfId, new TShardActorPrivateEvents::TEvResolveShard{});
                }

                event->Statuses.resize(pr.Statuses.size());
                for (auto& s: event->Statuses) {
                    s = {UrlStatusType::UNKNOWN_ERROR, errorMsg, 0};
                }
                actorSystemPtr->Send(new IEventHandle(hId, selfId, event.release(), 0, cookie));
            } else {
                event->Statuses = std::move(pr.Statuses);
                actorSystemPtr->Send(new IEventHandle(hId, selfId, event.release(), 0, cookie));
            }
        });
    }

    void OnUpdateConfig(const TShardActorEvents::TUpdateConfig::TPtr& request) {
        auto config = request->Get()->Config;
        Interval_ = GetInterval(config.Interval, IsYasmShard_);
        Shard_.UpdateConfig(config);
    }

    void OnAllDataProcessed(const TShardActorPrivateEvents::TEvOnAllDataProcessed::TPtr&) {
        MON_TRACE(ShardActor, "All interval data processed\n");

        ui8 sending = WritingAggregates_.exchange(true);

        if (!sending) {
            LabelPool_.Update();
        }

        auto f = Shard_.ProcessAggregates(CurrentWindowStart_, EndOfCurrentWindow_ - CurrentWindowStart_).Apply([this](auto f) {
            WritingAggregates_.store(false);
            return f;
        });

        SubscribeToProcessedData(f, TActorId(), 0, 1)
            .Subscribe([as = TActorContext::ActorSystem(), self = SelfId(), closed = IsShardClosed_](auto)
        {
            if (closed) {
                as->Send(self, new TEvents::TEvPoison);
            }
        });

        if (!IsShardClosed_) {
            StartInterval(EndOfCurrentWindow_);
        }
    }

    void OnPoison(const TEvents::TEvPoison::TPtr&) {
        if (!IsShardClosed_) {
            IsShardClosed_ = true;
        } else {
            PassAway();
        }
    }

    void InitTasksCounter() {
        TasksCounter_ = MakeHolder<TTasksCounter>([actorSystemPtr = NActors::TActorContext::ActorSystem(), self = SelfId()]() {
            actorSystemPtr->Send(self, new TShardActorPrivateEvents::TEvOnAllDataProcessed{});
        }, false);
    }

    static bool IsYasmShard(const TShardConfig& config) {
        // TODO: do not create YASM_PREFIX in both Shard and ShardActor
        static constexpr TStringBuf YASM_PREFIX = "yasm_";
        return config.Key.ProjectId.StartsWith(YASM_PREFIX);
    }

    void ReportLargeTs(TInstant ts) const {
        MON_WARN(
                ShardActor,
                "Document ts is too large. drop data. Info:"
                << "Interval(ms):" << Interval_.MilliSeconds()
                << ", current window start (ts):" << CurrentWindowStart_.MilliSeconds()
                << ", documents ts:" << ts.MilliSeconds());
    }

    size_t SizeBytes() {
        size_t res = sizeof(*this);
        if (TasksCounter_) {
            res += sizeof(*TasksCounter_);
        }
        res += ShardKey_.ProjectId.capacity();
        res += ShardKey_.ClusterName.capacity();
        res += ShardKey_.ServiceName.capacity();
        res += Shard_.SizeBytes() - sizeof(Shard_);

        for (const auto& windowQueue: NextWindowQueue_) {
            res += windowQueue.SizeBytes();
        }
        for (const auto& skipWindowQueue: SkipWindowQueue_) {
            res += skipWindowQueue.SizeBytes();
        }

        Y_VERIFY(res < (1l << 34));
        return res;
    }

private:
    TLabelPool LabelPool_;
    TShardId ShardNumId_;
    TString ShardStrId_;
    THolder<TTasksCounter> TasksCounter_;
    std::shared_ptr<IClusterRpc<NMemStore::IMemStoreRpc>> MemStoreRpc_;
    TActorId MemStoreClusterWatcher_;
    NMemStore::IMemStoreRpc* MemStoreNode_{nullptr};
    std::shared_ptr<TShardMeteringContext> ShardMeteringContext_;

    bool IsYasmShard_;
    TDuration Interval_;
    TInstant CurrentWindowStart_;
    TInstant EndOfCurrentWindow_;
    TInstant EndOfNextWindow_;
    TInstant EndOfSkipWindow_;

    TShardMetrics Metrics_;
    TShardKey ShardKey_;
    TShard Shard_;
    // TODO: optimization -- compute an actual user time
    TSteadyTimer Timer_;

    bool IsWindowClosed_;
    bool IsShardClosed_{false};
    TIntrusiveListWithAutoDelete<TDocumentData, TDelete> NextWindowQueue_;
    TIntrusiveListWithAutoDelete<TDocumentData, TDelete> SkipWindowQueue_;

    std::atomic<ui8> WritingAggregates_{false};

    constexpr static const TDuration DEFAULT_INTERVAL = TDuration::Seconds(15);
};

} // namespace

NActors::IActor* CreateShardActor(
    TShardConfig shardConfig,
    IThreadPool* processorsExecutor,
    std::shared_ptr<IClusterRpc<NMemStore::IMemStoreRpc>> memStoreRpc,
    TActorId memStoreClusterWatcher,
    std::shared_ptr<TShardMeteringMetricsRepository> meteringMetricsRepo,  // NOLINT(performance-unnecessary-value-param): false positive
    const TString& yasmPrefix,
    NMonitoring::IMetricRegistry* registry)
{
    return new TShardActor(
            std::move(shardConfig),
            processorsExecutor,
            std::move(memStoreRpc),
            memStoreClusterWatcher,
            std::move(meteringMetricsRepo),
            yasmPrefix,
            registry);
}

} // namespace NSolomon::NIngestor
